#
# Dark Channel Channel Storage Utility Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Utils::ChannelStorage;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use DarkChannel::Utils::Log;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_channel_create
                  dc_channel_destroy
                  dc_channel_data_get
                  dc_channel_data_set
                  dc_channel_data_delete
                  dc_channel_all
                  dc_channel_all_sid
                  dc_channel_find
                  dc_channel_exists );

# channel state indexed by $channel
my $CHANNELS = {};

#
# create channel with given name
#
sub dc_channel_create($$)
{
    my $channel = shift;
    my $sdata = shift;

    $CHANNELS->{$channel} = $sdata;
}

#
# destroy channel with given name
#
sub dc_channel_destroy($)
{
    my $channel = shift;
    my $sdata = $CHANNELS->{$channel};

    return delete $CHANNELS->{$channel};
}

#
# gets data of channel <channel>
#
# $channels = dc_channel_data_get()
# $channel = dc_channel_data_get($channel)
# $val = dc_channel_data_get($channel, $key)
# $val = dc_channel_data_get($channel, $key, $key2)
# $val = dc_channel_data_get($channel, $key, $key2, $key3)
#
sub dc_channel_data_get(;$$$$)
{
    my $channel = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;

    return $CHANNELS if (not $channel);
    return $CHANNELS->{$channel} if (not $key);
    return $CHANNELS->{$channel}->{$key} if (not $key2);
    return $CHANNELS->{$channel}->{$key}->{$key2} if (not $key3);
    return $CHANNELS->{$channel}->{$key}->{$key2}->{$key3};
}

#
# sets data of channel <channel>
#
# $channels = dc_channel_data_set()
# $channels = dc_channel_data_set($channels)
# $newval = dc_channel_data_set($channel, $newval)
# $newval = dc_channel_data_set($channel, $key, $newval)
# $newval = dc_channel_data_set($channel, $key, $key2, $newval)
#
sub dc_channel_data_set(;$$$$)
{
    my $channel = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;

    return ($CHANNELS = {}) if (not $channel);
    return ($CHANNELS = $channel) if (not $key);
    return ($CHANNELS->{$channel} = $key) if (not $key2);
    return ($CHANNELS->{$channel}->{$key} = $key2) if (not $key3);
    return ($CHANNELS->{$channel}->{$key}->{$key2} = $key3);
}

#
# delete data of channel <channel>
#
# $oldchannels = dc_channel_data_delete()
# $oldchannel = dc_channel_data_delete($channel)
# $oldval = dc_channel_data_delete($channel, $key)
# $oldval = dc_channel_data_delete($channel, $key, $key2)
# $oldval = dc_channel_data_delete($channel, $key, $key2, $key3)
#
sub dc_channel_data_delete(;$$$$)
{
    my $channel = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;
    my $channels = $CHANNELS;

    $CHANNELS = {}, return $channels if (not $channel);
    return delete $CHANNELS->{$channel} if (not $key);
    return delete $CHANNELS->{$channel}->{$key} if (not $key2);
    return delete $CHANNELS->{$channel}->{$key}->{$key2} if (not $key3);
    return delete $CHANNELS->{$channel}->{$key}->{$key2}->{$key3};
}

#
# returns 1 if channel <channel> exists
#
sub dc_channel_exists($)
{
    my $channel = shift;
    return (dc_channel_data_get($channel) ? 1 : 0)
}

#
# returns an array of all channels
#
sub dc_channel_all()
{
    return keys %{$CHANNELS};
}

#
# returns an array of all channels where <sid> is member
#
sub dc_channel_all_sid($)
{
    my $sid = shift // confess("dc_channel_all_sid() without pattern!");
    my @found = ();

    # find all channel names where <sid> is member
    for my $channel (keys %{$CHANNELS}) {
        push(@found, $channel) if (defined($CHANNELS->{$channel}->{members}->{$sid}));
    }

    return @found;
}

#
# returns an array of all channels matching <pattern>
#
sub dc_channel_find($)
{
    my $pattern = shift // confess("dc_channel_find() without pattern!");

    my @channels = keys %{$CHANNELS};
    return grep(/$pattern/, @channels);
}
