#
# Dark Channel Protocol V1 Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::V1;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use File::Path;
use POSIX qw(strftime);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_proto_transition
                  dc_proto_version );

#
# Channel Server State Machine
#
# <state> =>
# {
#    <input> => [ <output>, <next-state> ],
#    <input> => [ <output>, <next-state> ],
# }
#

my $DC_STATES = {

    'v1' => {
        'ChannelServer-Listener' =>
        {
            'initialization' => {
                always               => [ 'WELCOME', 'welcome-sent' ],
            },

            'welcome-sent' => {
                'HELLO'              => [ 'HELLO', 'connected' ],
                else                 => [ 'NOP', 'disconnected' ],
            },

            'connected' => {
                'JOIN'               => [ 'JOIN', 'connected' ],
                'PART'               => [ 'PART', 'connected' ],
                'RELAY'              => [ 'RELAY', 'connected' ],
                'PING'               => [ 'PONG', 'connected' ],
                'LIST'               => [ 'LIST', 'connected' ],
                'REGISTER'           => [ 'REGISTER', 'connected' ],
                'CHANNELSERVER'      => [ 'NOP', 'connected' ],
                else                 => [ 'NOP', 'disconnected' ],
            },

            'disconnected' => {         # end state
                else                 => [ 'NOP', 'disconnected' ],
            },
        },

        'ChannelServer-Connector' =>
        {
            'initialization' => {
                'WELCOME'            => [ 'HELLO', 'welcome-received' ],
                else                 => [ 'NOP', 'disconnected' ],
            },

            'welcome-received'       => {
                'HELLO'              => [ 'NOP', 'connected' ],
               else                  => [ 'NOP', 'disconnected' ],
            },

            'connected' => {
                'JOIN'               => [ 'NOP', 'connected' ],
                'PART'               => [ 'NOP', 'connected' ],
                'PONG'               => [ 'NOP', 'connected' ],
                'LIST'               => [ 'NOP', 'connected' ],
                'REGISTER'           => [ 'NOP', 'connected' ],
                'RELAY'              => [ 'NOP', 'relay-received' ],

                'cmd_JOIN'           => [ 'JOIN', 'connected' ],
                'cmd_PART'           => [ 'PART', 'connected' ],
                'cmd_PING'           => [ 'PING', 'connected' ],
                'cmd_LIST'           => [ 'LIST', 'connected' ],
                'cmd_REGISTER'       => [ 'REGISTER', 'connected' ],
                'cmd_CHANNELSERVER'  => [ 'CHANNELSERVER', 'connected' ],
                'cmd_RELAY_MESSAGE'  => [ 'RELAY_MESSAGE', 'connected' ],
                'cmd_RELAY_NICK'     => [ 'RELAY_NICK', 'connected' ],
                else                 => [ 'NOP', 'disconnected' ],
            },

            'relay-received' => {
                'MESSAGE'            => [ 'NOP', 'connected' ],
                'NICK'               => [ 'NOP', 'connected' ],
            },

            'disconnected' => {         # end state
                else                 => [ 'NOP', 'disconnected' ],
            },

        }
    },
};

#
# return next state for given <protocol-version>, <service-name>, <current-state>, <input-msg>
#
sub dc_proto_transition($$$$)
{
    my $protocol_version = shift // confess("Missing protocol_version!");
    my $service_name = shift // confess("Missing service_name!");
    my $current_state = shift // confess("Missing current_state!");
    my $input_msg = shift // confess("Missing input_msg!");

    my $result = $DC_STATES->{$protocol_version}->{$service_name}->{$current_state}->{$input_msg}
                 // $DC_STATES->{$protocol_version}->{$service_name}->{$current_state}->{else};

    confess("Failed to find transition: STATE->{$protocol_version}->{$service_name}->{$current_state}->{$input_msg}!") if (not $result);

    return @{$result};
}

#
# returns 1 if <protocol-version> is supported
#
sub dc_proto_version($)
{
    my $protocol_version = shift // confess("Missing protocol_version!");

    return ($DC_STATES->{$protocol_version}) ? 1 : 0
}

1;
