#
# Dark Channel Server Request Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Server::Request;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::Block;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::V1;

use DarkChannel::Node::ChannelServer::Conf;

use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_server_request_initialize
                  dc_server_request_process );

my $CONSUMER = '';

#
# import public key material after inspecting it and verify the signature
#
# returns: ($signature_state, $data [, $lsign])
#
#          signature_state: ['valid/known', 'valid/unknown', 'invalid']
#          data:            signed data
#
sub dc_server_request_import($$;$)
{
    my $sid = shift;
    my $pubkey = shift // confess('no public key!');
    my $lsign = shift // 0;

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');

    # XXX: IDEA: implement option to disallow unknown client connections effectively
    #            disallowing anyone new (and so with a unknown public key) from connecting
    dc_log_info("unknown client public key received, eventually a new and unknown client?", $alias);
    dc_log_info("importing new client's public key and re-validating received request", $alias);

    # inspect pubkey and remember keyid
    my $inspect = crypt_base_key_inspect($pubkey); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received request containing more than 1 public key!", $alias);
        return 0;
    }
    unless ($inspect->{$keyids[0]}->{key_type} eq 'pub') {
        dc_log_warn("received request containing a non 'pub' key!", $alias);
        return 0;
    }
    my $pubkey_id = $keyids[0];

    # import the armored key if it's a single public key
    my $import = crypt_base_key_import($pubkey);
    my $keyid = 0;

    # check if import succeeded
    return 0 unless ($import);

    # fetch imported keyid
    @keyids = keys %{ $import };
    $keyid = $keyids[0];

    # lsign the received key if requested
    my $export = crypt_base_key_sign($key_id, $pubkey_id, 'exportable') if ($lsign);

    return $keyid;
}

#
# detach last block, assume it's a signature and verify the signature
#
# returns: ($signature_state, $data)
#
#          signature_state: ['valid/known', 'valid/unknown', 'invalid']
#          data:            signed data
#
sub dc_server_request_verify($$)
{
    my ($sid, $request) = @_;

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');

    # log request if requested
    dc_log_dbg($request, $alias . ': Request') if ($CONF->{log}->{log_dbg_request});

    # fail if no request
    return(1, '') unless($request);

    # split signature from blob (the last block needs to be the signature, all requests are signed)
    my ($data, $signature) = dc_block_split_last($CONF, $request);

    # check signature
    my $sig = crypt_base_data_verify($signature, $data, $key_id);
    my $sig_state = $sig->{state} // 'invalid';
    my $sig_keyid = $sig->{key_id} // 0;

    dc_log_dbg(Dumper($sig), $alias . ': Signature') if ($CONF->{log}->{log_dbg_verify});
    return ($sig_state, $data);
}

#
# decrypt request and verify the signature
#
# returns: ($signature_state, $data)
#
#          signature_state: ['valid/known', 'valid/unknown', 'invalid']
#          data:            decrypted data
#
sub dc_server_request_decrypt_and_verify($$)
{
    my $sid = shift;
    my $request = shift // '';

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');

    # log request if requested
    dc_log_dbg($request, $alias . ': Request Transport') if ($CONF->{log}->{log_dbg_request_transport});

    # decrypt and check signature
    my $decrypted = crypt_base_data_decrypt($request, $key_id);

    # inspect result
    my $signature_state = $decrypted->{signature}->{state} // 'invalid';
    my $signature_keyid = $decrypted->{signature}->{key_id} // 0;

    my ($nickname_state, $nickname_keyid) = ('invalid', 0);
    $nickname_state = $decrypted->{nickname}->{state} // 'invalid' if ($decrypted->{nickname});
    $nickname_keyid = $decrypted->{nickname}->{key_id} // 0 if ($decrypted->{nickname});

    # log result
    if ($CONF->{log}->{log_dbg_verify}) {
        if ($signature_state eq 'valid/known') {
            dc_log_dbg('encrypted request has been successfully decrypted and verified (signer_keyid='
                       . $signature_keyid . ')', $alias . ': Signature');
        }
        elsif ($signature_state eq 'valid/unknown') {
            dc_log_dbg('encrypted request has been successfully decrypted and but signer is unknown to us!',
                       $alias . ': Signature');
        }
        else {
            dc_log_dbg('request could not be decrypted!',  $alias . ': Signature');
        }
    }

    # log decrypted request if requested
    dc_log_dbg($decrypted->{data}, $alias . ': Request') if ($CONF->{log}->{log_dbg_request});

    return ($decrypted->{data}, $signature_keyid, $signature_state, $nickname_keyid, $nickname_state);
}

sub dc_server_request_HELLO($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $pubkey = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # inspect pubkey and remember keyid
    my $inspect = crypt_base_key_inspect($pubkey); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received 'HELLO' request containing more than 1 key!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{key_type} ne 'pub') {
        dc_log_warn("received 'HELLO' request containing a non 'pub' key!", $alias);
        return 0;
    }
    my $pubkey_id = $keyids[0];

    # verify protocol specification: command arguments
    my $protocol_version;
    if ($arg =~ /^$CONF->{product_name}: Client: (v[0-9]+): [^\s]+$/) {
        $protocol_version = $1;
    }
    else {
        dc_log_warn("received 'HELLO' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }
    # verify protocol specification: check protocol version
    unless (dc_proto_version($protocol_version))
    {
        dc_log_warn("received 'HELLO' request requesting protocol version '" . $protocol_version .
                    "' but we do not support that version, ignoring request!", $alias);
        return 0;
    }

    # consume HELLO
    return dc_server_request_consume($sid, $cmd, $arg, $pubkey, $pubkey_id);
}

sub dc_server_request_JOIN($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless($arg =~ /^#[a-zA-Z-_]+$/)
    {
        dc_log_warn("received 'JOIN' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # consume JOIN
    return dc_server_request_consume($sid, $cmd, $arg);
}

sub dc_server_request_PART($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless($arg =~ /^#[a-zA-Z-_]+$/)
    {
        dc_log_warn("received 'PART' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # consume PART
    return dc_server_request_consume($sid, $cmd, $arg);
}

sub dc_server_request_RELAY($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless(($arg =~ /^#[a-zA-Z-_]+$/) || ($arg =~ /^0x[0-9A-F]+$/))
    {
        dc_log_warn("received 'RELAY' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # verify protocol specification: encrypted data
    unless($blob)
    {
        dc_log_warn("received 'RELAY' request without encrypyed data, ignoring request!", $alias);
        return 0;
    }

    # consume RELAY
    return dc_server_request_consume($sid, $cmd, $arg, $blob);
}

sub dc_server_request_PING($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless($arg =~ /^\d+$/)
    {
        dc_log_warn("received 'PING' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # consume PING
    return dc_server_request_consume($sid, $cmd, $arg);
}

sub dc_server_request_LIST($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless(($arg eq '') || ($arg =~ /^(#?[a-zA-Z0-9-]+)$/))
    {
        dc_log_warn("received 'LIST' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # consume PING
    return dc_server_request_consume($sid, $cmd, $arg);
}

sub dc_server_request_REGISTER($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');
    my $fqdn = `hostname -f`; chomp($fqdn);
    my ($role, $name);

    # verify protocol specification: command arguments
    if($arg =~ /^([A-Z]+) ([a-zA-Z-_]+)$/) {
        if ($1 ne 'NICKNAME') {
            dc_log_warn("received not supported 'REGISTER $1' request, ignoring request!", $alias);
            return 0;
        }
        $role = $1;
        $name = $2;
    }
    else {
        dc_log_warn("received 'REGISTER' request with invalid argument, ignoring request!", $alias);
        return 0;
    }

    # verify protocol specification: public key
    unless($blob)
    {
        dc_log_warn("received 'REGISTER " . $role . "' request without public key material, ignoring request!",
                    $alias);
        return 0;
    }

    # inspect pubkey and remember keyid
    my $inspect = crypt_base_key_inspect($blob); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received 'REGISTER " . $role . "' request containing more than 1 key!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{key_type} ne 'pub') {
        dc_log_warn("received 'REGISTER " . $role . "' request containing a non 'pub' key!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{uid_comment} ne ucfirst(lc($role))) {
        dc_log_warn("'REGISTER " . $role . "' validation failed: key with not supported comment!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{uid_name} ne $CONF->{product_name} . ' ' . ucfirst(lc($role))) {
        dc_log_warn("'REGISTER " . $role . "' validation failed: key with not supported name!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{uid_email} !~ /^$name\@$fqdn/) {
        dc_log_warn("'REGISTER " . $role . "' validation failed: key with not supported email!", $alias);
        return 0;
    }
    my $pubkey = $blob;
    my $pubkey_id = $keyids[0];

    # consume REGISTER
    return dc_server_request_consume($sid, $cmd, $arg, $role, $name, $pubkey, $pubkey_id);
}

sub dc_server_request_CHANNELSERVER($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless (($arg eq 'DUMP SESSION')
        || ($arg eq 'DUMP CHANNEL')
        || ($arg eq 'DUMP CRYPTO')
        || ($arg eq 'DUMP CONF'))
    {
        dc_log_warn("received 'CHANNELSERVER' request with invalid arguments, ignoring request!", $alias);
        return 0;
    }

    # consume CHANNELSERVER
    return dc_server_request_consume($sid, $cmd, $arg);
}


#
# send consume event to consumer
#
# dc_client_request_consume($sid, $cmd, $arg, ...)
#

sub dc_server_request_consume($$@)
{
    my ($sid, $cmd, $arg) = (shift, shift, shift);
    my $params = \@_;
    my $event = 'consume_' . $cmd;

    $poe_kernel->post($CONSUMER, $event, $sid, $arg, $params)
        || confess("failed to send event '" . $event . "' to consumer '" . $CONSUMER . "'!");
    return 1;
}

#
# process received client request
#
sub dc_server_request_process($$)
{
    my ($sid, $request) = @_;

    my $transport_encryption = $CONF->{settings}->{transport_encryption} // 1;
    my ($data, $sig_keyid, $sig_state, $nick_keyid, $nick_state) = ('', 0, 'invalid', 0, 'invalid');

    # mark client as seen
    my $now = strftime "%Y-%m-%d-%H%M", localtime;
    dc_session_data_set($sid, 'time_seen', $now);

    # decrypt and verify or verify
    if ($transport_encryption) {
        # this clients signature key has to be known and decryption & signature verification has to succeed
        ($data, $sig_keyid, $sig_state, $nick_keyid, $nick_state)
            = dc_server_request_decrypt_and_verify($sid, $request);
    }
    else {
        # this clients signature key has to be known and signature verification has to succeed
        ($sig_state, $data) = dc_server_request_verify($sid, $request);
    }

    # parse command line of received request
    my @lines = split(/\n/, $data);
    my $alias = dc_session_data_get($sid, 'alias');
    my ($cmd, $arg, $blob) = ('', '', '');

    if ($lines[0] && (($lines[0] =~ /^([A-Z]+)$/) || ($lines[0] =~ /^([A-Z]+) (.*)$/))) {
        ($cmd, $arg) = ($1, $2);
        shift @lines;
        $blob = join("\n", @lines);
    }
    else {
        dc_log_warn("request command is not well formed [A-Z]!", $alias);
        return 0;
    }

    # HELLO is the only command allowed to be signed by an unknown entity, for all others verify has to succeed
    unless($sig_state eq 'valid/known') {
        # do not fail if HELLO message is not verified
        unless(($cmd eq 'HELLO') && $blob && ($sig_state eq 'valid/unknown')) {
            dc_log_warn("received request with invalid signature (state=" . $sig_state . "), ignoring request!",
                        $alias);
            return 0;
        }

        # if this is a client connecting the first time, he will sign with an unknown public key
        # and so signature verification will fail. handle this gracefully by importing and signing
        # the public key locally prior to re-validating the request
        my $lsign = 1;
        my $keyid = dc_server_request_import($sid, $data, $lsign);

        # re-validate
        if ($transport_encryption) {
            # this clients signature key has to be known and decryption & signature verification has to succeed
            ($data, $sig_keyid, $sig_state, $nick_keyid, $nick_state)
                = dc_server_request_decrypt_and_verify($sid, $request);
        }
        else {
            # this clients signature key has to be known and signature verification has to succeed
            ($sig_state, $data) = dc_server_request_verify($sid, $request);
        }
        unless($sig_state eq 'valid/known') {
            dc_log_warn("received 'HELLO' request with invalid signature (state=" . $sig_state
                        . "), ignoring request!", $alias);
            return 0;
        }
    }

    # call SERVER REQUEST function (CONSUMER)
    my $request_func_name = 'dc_server_request_' . $cmd;
    dc_log_dbg("'" . $request_func_name . "()'", , $alias . ': Request: Call Consumer')
        if ($CONF->{log}->{log_dbg_transition});

    # check if request function exists and call it
    if (DarkChannel::Proto::Server::Request->can($request_func_name)) {
        # XXX: TODO: dont confess() but react accordingly on failed request intergrity check
        my $request_func = \&$request_func_name;
        my $success = $request_func->($sid, $cmd, $arg, $blob, $sig_keyid, $nick_keyid);
        unless($success) {
            dc_log_warn("request integrity check failed in '$request_func_name'!");
            return 0;
        }
        return 1;
    }
    dc_log_warn("request function'" . $request_func_name . "()' does not exist!", $alias);
    return 0;
}

sub dc_server_request_initialize(;$)
{
    $CONSUMER = shift // 'ChannelServer-Interpreter';

    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Server::Request (consumer=" . $CONSUMER . ")");

    return 1;
}

1;
