#
# Dark Channel Server Protocol Interpreter Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Server::Interpreter;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::Text;
use DarkChannel::Utils::SessionStorage;
use DarkChannel::Utils::ChannelStorage;

use DarkChannel::Proto::V1;
use DarkChannel::Proto::Server::Request;
use DarkChannel::Proto::Server::Response;

use DarkChannel::Node::ChannelServer::Conf;

# POE Debugging
#sub POE::Kernel::ASSERT_DEFAULT () { 1 }
#sub POE::Kernel::ASSERT_EVENTS  () { 1 }
#sub POE::Kernel::CATCH_EXCEPTIONS () { 0 }

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_server_interpreter_initialize
                  dc_server_interpreter_spawn );

sub i_log_info($@)
{
    dc_log_info($_[0], $_[1]) if ($CONF->{log}->{log_dbg_interpreter});
}

sub i_log_warn($@)
{
    dc_log_warn($_[0], $_[1]) if ($CONF->{log}->{log_dbg_interpreter});
}

sub i_log_err($@)
{
    dc_log_err($_[0], $_[1]) if ($CONF->{log}->{log_dbg_interpreter});
}

sub i_log_crit($@)
{
    dc_log_crit($_[0], $_[1]) if ($CONF->{log}->{log_dbg_interpreter});
}

sub i_log_dbg($@)
{
    dc_log_dbg($_[0], $_[1]) if ($CONF->{log}->{log_dbg_interpreter});
}

#
# join a new client <cpubkey> to a channel <channel>
#
sub dc_server_interpreter_channelmember_join($$)
{
    my $sid = shift;
    my $channel = shift;

    # check if member already is on channel
    return undef if (dc_channel_data_get($channel, 'members', $sid));

    # channel dataset
    my $now = strftime "%Y-%m-%d-%H%M", localtime;
    my $cdata = {
        creation => {
            time_founding => $now,
            founder => $sid,         # XXX: TODO: use keyid as founder
        },
        members => {
            $sid => {
                time_join => $now,
                time_pong => time(),
                alias => dc_session_data_get($sid, 'alias'),
                sap_direct => 0,
            },
        },
    };

    # add member for new channel
    if (not dc_channel_exists($channel))
    {
        # create new dataset for channel
        i_log_info("creating new channel '" . $channel . "'", 'ChannelStorage');
        dc_channel_create($channel, $cdata)
    }
    # add member for existing channel
    else
    {
        # add new member to existing dataset
        dc_log_dbg("adding new client to channel '" . $channel . "'", 'ChannelStorage');
        dc_channel_data_set($channel, 'members', $sid, $cdata->{members}->{$sid});
    }

    # store channel name in user session
    dc_session_data_set($sid, 'channels', $channel, $now);

    return $cdata->{members}->{$sid};
}

#
# part client <cpubkey> from a channel <channel>
#
sub dc_server_interpreter_channelmember_part($$)
{
    my $sid = shift;
    my $channel = shift;

    # check if member already is not on channel
    return undef unless (dc_channel_data_get($channel, 'members', $sid));

    # count members on channel
    my $members = dc_channel_data_get($channel, 'members');
    my $count = $members ? (keys %{ $members }) : 0;

    # remove member from channel
    my $ochannel;
    if ($count eq 1)
    {
        # remove last member of channel -> close channel
        i_log_info("removing last client from channel '".$channel."' and destroying channel", 'ChannelStorage');
        i_log_info("destroying channel '" . $channel . "'", 'ChannelStorage');
        $ochannel = dc_channel_destroy($channel);
    }
    else
    {
        # remove member from dataset -> leave channel opoen
        dc_log_dbg("removing client from channel '" . $channel . "'", 'ChannelStorage');
        $ochannel = delete $members->{$sid};
    }

    # remove channel name from user session
    dc_session_data_delete($sid, 'channels', $channel);

    return $ochannel;
}

sub dc_server_interpreter_statetransition($$$)
{
    my $state_machine = shift;
    my $sid = shift;
    my $cmd = shift;

    my $protocol_version = dc_session_data_get($sid, 'protocol_version') // 'v1';
    my $proto_state = dc_session_data_get($sid, 'proto_state');
    my $alias = dc_session_data_get($sid, 'alias');

    # Fetch transition from transition matrix
    my ($response, $state) = dc_proto_transition($protocol_version, $state_machine, $proto_state, $cmd);

    my $request_response = '[' . $cmd . ' / ' . $response . ']';
    my $state_change  = $proto_state . '/' . $cmd . ' -> ' . $state . '/' . $response;
    dc_log_dbg($state_change, $alias . ': Protocol: ' . $protocol_version)
        if ($CONF->{log}->{log_dbg_transition});

    # change current client's state
    dc_log_dbg($state, $alias . ': Protocol: ' . $protocol_version . ': State')
        if ($CONF->{log}->{log_dbg_transition});
    dc_session_data_set($sid, 'proto_state', $state);

    return ($response, $state);

}

sub dc_server_interpreter_spawn($;$)
{
    my $state_machine = shift // $CONF->{service_name};
    my $alias_interpreter = shift // 'ChannelServer-Interpreter';
    my $debug = $CONF->{log}->{log_dbg_session_interpreter};

    my $session = POE::Session->create(

        options => { debug => $debug, trace => 0, default => 1 },

        inline_states => {
            _start => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # raport startup
                dc_log_dbg("channel server interpreter session created", $alias_interpreter);

                # set alias and store alias on heap
                $kernel->alias_set($alias_interpreter);
                $heap->{alias} = $alias_interpreter;

                # store state machine
                $heap->{state_machine} = $state_machine;
            },

            _stop => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
            },

            consume_HELLO => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($pubkey, $pubkey_id) = @{$params};
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                i_log_dbg("consuming 'HELLO' event from client (sid=" . $sid . ")", $alias);

                # consume HELLO: choose protocol version and store informations in session
                if ($arg =~ /^$CONF->{product_name}: ([^\s]+): (v[0-9]+): ([^\s]+)$/)
                {
                    my ($service_type, $proto_version, $node_type) = ($1, $2, $3);

                    # choose minimal protocol version between client and channel server
                    my $proto_server = $CONF->{protocol_version};
                    $proto_version = ($proto_version lt $proto_server) ? $proto_version : $proto_server;

                    # store data in session
                    dc_session_data_set($sid, 'client', 'protocol_version', $proto_version);
                    dc_session_data_set($sid, 'client', 'service_type', $service_type);
                    dc_session_data_set($sid, 'client', 'node_type', $node_type);
                }
                dc_session_data_set($sid, 'client', 'key_id', $pubkey_id) if ($pubkey_id);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'HELLO');

                # send response if not NOP
                $kernel->yield('send_response', $sid, $response) if ($response ne 'NOP');
            },

            consume_JOIN => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $channel = $arg;
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                # consume JOIN: add new member to channel
                i_log_dbg("consuming 'JOIN' event from client (sid=" . $sid . ")", $alias);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'JOIN');

                # send response if JOIN
                if ($response eq 'JOIN') {
                    # check if there are members on the channel
                    my $members = dc_channel_data_get($channel, 'members');
                    if ($members && (keys %{ $members })) {
                        # check if client is already member of channel and fail if so
                        if ($members->{$sid}) {
                            i_log_err("received 'JOIN' request while client is already member of channel '" .
                                       $channel . "', ignoring request", $alias);
                            return;
                        }

                        # send JOIN to new client informing him of all channel members
                        my @ckeyid = ();
                        for my $csid (keys %{ $members }) {
                            # fetch client keyid
                            push(@ckeyid, dc_session_data_get($csid, 'client', 'key_id'));
                        }
                        my $cpubkey = crypt_base_key_export(\@ckeyid);
                        i_log_dbg("sending 'JOIN ".$channel."' response to new client containing "
                                  . ($#ckeyid + 1) . " public key(s) " . join(', ', @ckeyid), $alias);
                        $kernel->yield('send_response', $sid, $response, [ $channel, $cpubkey ]);

                        # send JOIN to all channel members informing them of the new client
                        my $pubkeyid = dc_session_data_get($sid, 'client', 'key_id');
                        my $pubkey = crypt_base_key_export($pubkeyid);
                        for my $csid (keys %{ $members }) {
                            i_log_dbg("sending 'JOIN " . $channel . "' response to channel member client (sid="
                                      . $csid . ")", $alias);
                            $kernel->yield('send_response', $csid, $response, [ $channel, $pubkey ]);
                        }
                    }
                    else {
                        # no prior channel members, inform client on JOIN
                        i_log_dbg("sending 'JOIN " . $channel
                                  . "' response to new client (no prior channel members)", $alias);
                        $kernel->yield('send_response', $sid, $response, [ $channel ]);
                    }

                    # add new client to channel
                    dc_server_interpreter_channelmember_join($sid, $channel);
                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_JOIN()", $alias);
                }
            },

            consume_PART => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($channel) = $arg;
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                # consume PART: remove member from channel
                i_log_dbg("consuming 'PART' event from client (sid=" . $sid . ")", $alias);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'PART');

                # send response if PART
                if ($response eq 'PART') {
                    # check if there are members on the channel
                    my $members = dc_channel_data_get($channel, 'members');
                    if ($members) {
                        # check if client is member of channel and fail if not so so
                        unless ($members->{$sid}) {
                            i_log_err("received 'PART' request for channel '" . $channel .
                                      "' for member not on channel, ignoring request", $alias);
                            return;
                        }

                        # fetch keyid of client from session
                        my $keyid = dc_session_data_get($sid, 'client', 'key_id');

                        # send PART to parting client
                        i_log_dbg("sending 'PART " . $channel . " " . $keyid
                                  . "' response to parting client (sid=". $sid . ")", $alias);
                        $kernel->yield('send_response', $sid, $response, [ $channel, $keyid ]);

                        # remove client from channel
                        dc_server_interpreter_channelmember_part($sid, $channel);

                        # send PART to all channel members left on the channel
                        for my $csid (keys %{ $members }) {
                            i_log_dbg("sending 'PART " . $channel . " " . $keyid
                                      . "' response to channel member client (sid=". $csid . ")", $alias);
                            $kernel->yield('send_response', $csid, $response, [ $channel, $keyid ]);
                        }
                    }

                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_PART()", $alias);
                }
            },

            consume_RELAY => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($encrypted) = @{$params};
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                # consume RELAY: send encrypted content to all channel members
                i_log_dbg("consuming 'RELAY' event from client (sid=" . $sid . ")", $alias);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'RELAY');

                # send response if RELAY
                if ($response eq 'RELAY') {
                    if ($arg =~ /^#[a-zA-Z_-]+$/) {
                        # recipient is a channel
                        my $channel = $arg;

                        # check if members on channel
                        my $members = dc_channel_data_get($channel, 'members');
                        if ($members) {
                            # check if client is already member of channel and fail if so
                            unless ($members->{$sid}) {
                                i_log_err("received 'RELAY " . $channel
                                          . "' request from non member client, ignoring request", $alias);
                                return;
                            }

                            # send RELAY to all channel member clients
                            for my $csid (keys %{ $members }) {
                                # ignore RELAY sender
                                next if ($csid eq $sid);
                                # send RELAY
                                i_log_dbg("sending 'RELAY " . $channel
                                          . "' response to channel member client (sid=". $csid . ")", $alias);
                                $kernel->yield('send_response', $csid, $response, [ $channel, $encrypted ]);
                            }
                        }
                        else {
                            # no prior channel members, ignore RELAY
                            i_log_dbg("ignoring 'RELAY ".$channel."', there are no channel members", $alias);
                        }
                    }
                    else {
                        # recipient is a single client keyid
                        my $keyid = $arg;

                        # find client with keyid
                        my $found = 0;
                        for my $csid (keys %{ dc_session_data_get() }) {
                            if ($keyid eq dc_session_data_get($csid, 'client', 'key_id')) {
                                # send RELAY
                                i_log_dbg("sending 'RELAY " . $keyid
                                          . "' response to channel member client (sid=". $csid . ")", $alias);
                                $kernel->yield('send_response', $csid, $response, [ $keyid, $encrypted ]);
                                $found = 1;
                                last;
                            }
                        }
                        unless($found) {
                            # no client with keyid found, ignore RELAY
                            i_log_dbg("ignoring 'RELAY " . $keyid ."', no client with that keyid found", $alias);
                        }
                    }
                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_RELAY()", $alias);
                }
            },

            consume_PING => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $timestamp = $arg;
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                # consume PING: send PONG (client has been marked as seen in process_request event)
                i_log_dbg("consuming 'PING' event from client (sid=" . $sid . ")", $alias);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'PING');

                # send response if PONG
                if ($response eq 'PONG') {
                    # send PONG to requesting client
                    i_log_dbg("sending 'PONG " . $timestamp
                                  . "' response to pinging client (sid=". $sid . ")", $alias);
                    $kernel->yield('send_response', $sid, $response, [ $timestamp ]);
                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_PING()", $alias);
                }
            },

            consume_LIST => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $channel_pattern = $arg;
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};

                # consume PING: send PONG
                i_log_dbg("consuming 'LIST' event from client (sid=" . $sid . ")", $alias);

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'LIST');

                # send response if PONG
                if ($response eq 'LIST') {
                    # generate list of channels
                    my $channels = dc_channel_data_get();
                    my $list = {};
                    foreach my $channel (keys %{$channels}) {
                        next if ($channel_pattern && ($channel !~ /$channel_pattern/));
                        $list->{$channel} = {
                            name => $channel,
                            topic => '',
                        };
                    }

                    # send LIST to requesting client
                    i_log_dbg("sending 'LIST' response to pinging client (sid=". $sid . ")", $alias);
                    $kernel->yield('send_response', $sid, $response, [ $list ]);
                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_LIST()", $alias);
                }
            },

            consume_REGISTER => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($role, $name, $pubkey, $pubkey_id) = @{$params};
                my $namespace = lc($role);
                my $alias = dc_session_data_get($sid, 'alias');
                my $sm = $heap->{state_machine};
                my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');
                my $now = strftime "%Y-%m-%d-%H%M%S", localtime;
                my $pubkey_signed = '';

                # consume PING: send PONG (client has been marked as seen in process_request event)
                i_log_dbg("consuming 'REGISTER' event from client (sid=" . $sid . ", role=" . $role
                          . ", name=" . $name . ")", $alias);

                # check namespace and register if free
                unless (defined($CONF->{node}->{$namespace}->{$name})) {
                    # import, sign and export the public key
                    my $import = crypt_base_key_import($pubkey, $role);
                    $pubkey_signed = crypt_base_key_sign($key_id, $pubkey_id, 'exportable');

                    # register namespace
                    $CONF->{node}->{$namespace}->{$name} = $now;
                }

                # protocol state transition (-> next-state, response)
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'REGISTER');

                # send response if REGISTER
                if ($response eq 'REGISTER') {

                    # send REGISTER response to registering client
                    i_log_dbg("sending 'REGISTER " . $arg
                                  . "' response to registering client (sid=". $sid . ")", $alias);
                    $kernel->yield('send_response', $sid, $response, [ $role, $name, $pubkey_signed ]);
                }
                else {
                    i_log_err("response type '" . $response . "' not supported in consume_PING()", $alias);
                }
            },

            consume_CHANNELSERVER => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $cmd = $arg;
                my $alias = dc_session_data_get($sid, 'alias');

                # consume JOIN: add new member to channel
                i_log_dbg("consuming 'CHANNELSERVER' event from client (sid=".$sid.", cmd=".$cmd.")", $alias);

                if ($cmd =~ /^DUMP (.+)$/) {
                    my $subcmd = $1;

                    # collect & render debug data
                    my $data = '';
                    $data = Dumper(dc_session_data_get()) if ($subcmd eq 'SESSION');
                    $data = Dumper(dc_channel_data_get()) if ($subcmd eq 'CHANNEL');
                    $data = Dumper(crypt_base_data_get()) if ($subcmd eq 'CRYPTO');
                    $data = Dumper($CONF)                 if ($subcmd eq 'CONF');

                    if ($data) {
                        # replace VAR1 with sensible variable name for dump commands
                        my $key = uc($subcmd);
                        $data =~ s/VAR1/$key/g;

                        # shorten gpg data when dumping crypto data
                        $data = dc_text_transform_shorten_gpg_blocks($data);

                        # show data in log
                        i_log_info($data, $alias . ': DUMP: ' . ucfirst(lc($subcmd)));
                    }
                    else {
                        # unknown subcmd
                        i_log_err("ignoring 'CHANNELSERVER DUMP ".$subcmd."', the sub command is unknown", $alias);
                    }
                }
                else {
                    # unknown cmd
                    i_log_err("ignoring 'CHANNELSERVER ".$cmd."', the command is unknown", $alias);
                }


            },

            send_response => sub {
                my ($kernel, $heap, $session, $sid, $response, $response_args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];

                my $alias = dc_session_data_get($sid, 'alias');
                return unless($alias);

                dc_server_response_send($sid, $response, $response_args);
            },

            process_request => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $request) = @_[ARG0, ARG1];

                my $success = dc_server_request_process($sid, $request);
            },

            client_connected => sub {
                my ($kernel, $heap, $session, $sender, $alias, $sap, $sid, $keyid)
                    = @_[KERNEL, HEAP, SESSION, SENDER, ARG0, ARG1, ARG2, ARG3];
                my $sm = $heap->{state_machine};

                # new client connection
                dc_log_info("new client connected (sid=" . $sid . ")", $alias);

                # client session data
                my $now = strftime "%Y-%m-%d-%H%M", localtime;
                my $sdata = {
                    sap => $sap,
                    alias => $alias,
                    channelserver => { key_id => $keyid },
                    time_connect => $now,
                    time_seen => $now,
                    proto_state => 'initialization',
                };

                # register client's session
                dc_session_register($sid, $sdata);

                # use state machine to find initial state transition 'initialization/always'
                my ($response, $state) = dc_server_interpreter_statetransition($sm, $sid, 'always');

                # send initial state response if not NOP
                $kernel->yield('send_response', $sid, $response) if ($response ne 'NOP');
            },

            client_disconnected => sub {
                my ($kernel, $heap, $session, $alias, $sap, $sid)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];

                # fetch keyid of lost client from session
                my $keyid = dc_session_data_get($sid, 'client', 'key_id');

                # go through all channels the client is member of
                for my $channel (dc_channel_all_sid($sid)) {
                    # remove lost client from channel
                    dc_channel_data_delete($channel, 'members', $sid);

                    # send all remaining members of the channel a PART for the lost client
                    my $members = dc_channel_data_get($channel, 'members');
                    for my $csid (keys %{ $members }) {
                        i_log_dbg("sending 'PART ".$channel." " . $keyid .
                                  "' response to channel " . $channel . " member client (sid=". $csid . ")",
                                  $alias);
                        $kernel->yield('send_response', $csid, 'PART', [ $channel, $keyid ]);
                    }

                    # remove channel if no clients on channel anymore
                    unless ($members && (keys %{$members})) {
                        dc_log_info("no users on channel " . $channel . " anymore, destroying channel", 'Channel Storage');
                        dc_channel_destroy($channel);
                    }
                }

                # unregister client's session if still existing. forced channel server shutdown
                # leads to mass disconnecting/unregistering which can remove a session prior to
                # this event
                dc_session_unregister($sid) if (dc_session_data_get($sid));

                # lost client connection
                dc_log_info("client disconnected (sid=" . $sid . ")", $alias);
            },
        },
    );
}

sub dc_server_interpreter_initialize()
{
    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Server::Interpreter");

    return 1;
}

1;
