#
# Dark Channel Protocol Server Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Server;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use Getopt::Long;
use POSIX qw(strftime);

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::V1;
use DarkChannel::Proto::Server::Request;
use DarkChannel::Proto::Server::Response;
use DarkChannel::Proto::Server::Interpreter;

use DarkChannel::Node::ChannelServer::Conf;

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Component::Server::TCP);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 0.10;
our @ISA = qw( Exporter );
our @EXPORT = qw( dc_poe_server_initialize
                  dc_poe_server_spawn );

our @EXPORT_OK = qw();

# set by initialize
my $alias_interpreter;

sub dc_poe_server_process_request($$)
{
    my $sid = shift;
    my $heap = shift // confess("No heap in 'dc_poe_server_process_request'");
    my $alias = $heap->{alias};
    my $request = $heap->{input_buf}; chomp($request);

    # call interpreter process function
    if (not $poe_kernel->post($alias_interpreter, 'process_request', $sid, $request))
    {
        dc_log_err("failed to send post('" . $alias_interpreter . "' , 'process_request', " . $sid . ", ...)", $alias);
        dc_log_err("error message: $!", $alias);
    }

    # clear cmd
    $heap->{input_buf} = '';
}

sub dc_poe_server_process_input($$$)
{
    my ($sid, $heap, $input) = @_;
    my $alias = $heap->{alias} // 'unknown';

    if (not ($input =~ /^\.$/))
    {
        # cmd not yet finished
        dc_log_dbg("'$input'", $alias . ': Input') if ($CONF->{log}->{log_dbg_input});
        $heap->{input_buf} .= $input . "\n";
    }
    else
    {
        # cmd completely received, process cmd
        dc_poe_server_process_request($sid, $heap);
    }
}

#
# spawn channel server client listener session
#
sub dc_poe_server_spawn($$$)
{
    my ($listener_host, $listener_port, $key_id) = @_;

    my $alias_listener = 'ChannelServer-Listener-' . $listener_host . ':' . $listener_port;
    my $alias_client_template = 'Client-';

    my $debug = $CONF->{log}->{log_dbg_session_client_tcp};

    POE::Component::Server::TCP->new(
        Alias   => $alias_listener,
        Address => $listener_host,
        Port    => $listener_port,

        SessionParams => [ options => { debug => $debug, trace => 0, default => 1 } ],

        Started => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];

            # raport startup
            dc_log_dbg("listener session created", $alias_listener);

            # store alias
            $heap->{alias} = $alias_listener;

            # setup supported cryptosystems
            $key_id = crypt_base_setup_cryptosystems(undef, $key_id)
                or confess("failed to setup DarkChannel::Crypt::Base crypto systems!");

            # store key_id under the '_default' channel server name
            $CONF->{node}->{channelserver}->{_default}->{key_id} = $key_id;

            # raport listener
            my $sap = $listener_host . ':' . $listener_port;
            dc_log_info("listening on " . $sap);
        },

        ClientInput => sub {
            my ($kernel, $heap, $input, $session) = @_[KERNEL, HEAP, ARG0, SESSION];

            # process client input
            dc_poe_server_process_input($session->ID, $heap, $input);
        },

        ClientConnected => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
            my $sap =  $heap->{remote_ip} . ':' . $heap->{remote_port};
            my $sid = $session->ID;

            # store alias
            my $alias = $alias_client_template . $sap;
            $heap->{alias} = $alias;
            $kernel->alias_set($alias);

            # inform interpreter on new client connection
            $kernel->post($alias_interpreter, 'client_connected', $alias, $sap, $sid, $key_id);
        },

        ClientDisconnected => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
            my $sap =  $heap->{remote_ip} . ':' . $heap->{remote_port};
            my $sid = $session->ID;
            my $alias = $heap->{alias};

            # inform interpreter on lost client connection
            $kernel->post($alias_interpreter, 'client_disconnected', $alias, $sap, $sid);
        },

        # note to self: key 'InlineState' in POE::Component::Server::TCP go to into the
        # client session statemachine, so handling server side signals wont work here
        InlineStates => {
            _parent => sub {
                my ($kernel, $heap, $session, $parent_old, $parent_new) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my $alias = $heap->{alias};

                dc_log_crit("Session " . $_[SESSION]->ID . " parent changed from session " . $parent_old->ID .
                    " to session " . $parent_new->ID, $alias);
            },

            send => sub {
                my ($kernel, $heap, $session, $output) = @_[KERNEL, HEAP, SESSION, ARG0];

                # send received data to client
                $heap->{client}->put($output) if ($heap->{client});
            },
        },
    );
}

sub dc_poe_server_initialize(;$)
{
    # configure interpreter
    $alias_interpreter = shift // 'ChannelServer-Interpreter';

    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::ChannelServer::ClientListener");

    return 1;
}

1;
