#
# Dark Channel Client Request Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Client::Request;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX 'strftime';

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;
use DarkChannel::Utils::ChannelStorage;

use DarkChannel::Node::Client::Conf;

use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_client_request_initialize
                  dc_client_request_send );

sub dc_client_request($$)
{
    my ($sid, $request) = @_;

    my $transport_encryption = $CONF->{settings}->{transport_encryption} // 1;

    my $prefix_request = 'Request';
    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');
    my $recipient_key_id = dc_session_data_get($sid, 'server', 'key_id');
    my $request_transport;

    # check if transport encryption is enabled
    if ($transport_encryption && $recipient_key_id) {
        # add request end marker
        $request .= "\n.\n";

        # encrypt and sign request
        my $encrypted = crypt_base_data_encrypt($request, $key_id, $recipient_key_id);

        # log error if failed to sign
        unless ($encrypted) {
            my $err = "failed to to encrypt and sign request data, ignoring request!";
            dc_log_crit($err, $alias . ': ' . $prefix_request);
            dc_client_request_log($sid, $err, $prefix_request);
            return 0;
        }

        # replace request with encrytped and signed request
        $request_transport = $encrypted;
    }
    else {
        # no transport encryption: sign request
        my $signature = crypt_base_data_sign($request, $key_id);

        # log error if failed to sign
        unless ($signature) {
            my $err = "failed to add signature to request data, ignoring request!";
            dc_log_crit($err, $alias . ': ' . $prefix_request);
            dc_client_request_log($sid, $err, $prefix_request);
            return 0;
        }

        # combine request and signature
        my $prefix = ($request =~ m{\n}) ? "\n-" : '';
        $request_transport .= $prefix . "\n" . $signature;
    }

    # add request end marker
    $request_transport .= "\n.";

    return [ $request_transport, $request ];
}

sub dc_client_request_HELLO($$)
{
    my $sid = shift;
    my $args = shift;

    my @banner = (
        $CONF->{product_name},
        $CONF->{service_name},
        $CONF->{protocol_version},
        $CONF->{client_type}
    );
    my $key_id = dc_session_data_get($sid, 'key_id');
    my $pubkey = crypt_base_key_export_pub($key_id);
    my $request = 'HELLO' . ' ' . join(': ', @banner) . "\n" . $pubkey;

    return dc_client_request($sid, $request);
}

sub dc_client_request_JOIN($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel) = @{$args};

    confess("dc_client_request_JOIN(): no channel argument") if (not $channel);

    my $request = 'JOIN' . ' ' . $channel;

    return dc_client_request($sid, $request);
}

sub dc_client_request_PART($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel) = @{$args};

    confess("dc_client_request_PART(): no channel argument") if (not $channel);

    my $request = 'PART' . ' ' . $channel;

    return dc_client_request($sid, $request);
}

sub dc_client_request_MESSAGE($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel, $message) = @{$args};

    confess("dc_client_request_MESSAGE(): no channel argument") unless($channel);
    confess("dc_client_request_MESSAGE(): no message argument") unless($message);

    # terminate message with a newline
    $message .= "\n" unless ($message =~ /.*\n$/);

    my $request = 'MESSAGE' . ' ' . $channel . "\n" . $message;

    return dc_client_request($sid, $request);
}

sub dc_client_request_NICK($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel, $nickname_keyid) = @{$args};

    confess("dc_client_request_NICK(): no channel argument") unless($channel);
    confess("dc_client_request_NICK(): no nickname_keyid argument") unless($nickname_keyid);

    my $pubkey = crypt_base_key_export_pub($nickname_keyid);
    my $request = 'NICK' . ' ' . $channel . "\n" . $pubkey;

    return dc_client_request($sid, $request);
}

sub dc_client_request_RELAY($$)
{
    my $sid = shift;
    my $args = shift;
    my ($recipient, $data) = @{$args};

    confess("dc_client_request_RELAY(): no recipient argument") unless($recipient);
    confess("dc_client_request_RELAY(): no data argument") unless($data);

    my $key_id = dc_session_data_get($sid, 'key_id');
    my @signature_keyids = ($key_id);
    my (@recipient_keyids, $nickname_keyid);

    # ignore RELAY request if recipient is a channel and no members on channel
    if ($recipient =~ /^#/) {
        my $members = dc_channel_data_get($sid . ':' . $recipient, 'members');
        my $nickname = dc_channel_data_get($sid . ':' . $recipient, 'nickname');

        $nickname_keyid = $nickname ? $nickname->{key_id} : undef;
        @recipient_keyids = keys %{ $members };

        unless (@recipient_keyids) {
            dc_client_request_log($sid, 'will not RELAY to channel with no members!', 'Request');
            return 1;
        }
    }
    elsif ($recipient =~ /^0x/) {
        $nickname_keyid = undef; # XXX: TODO: fetch nickname keyid for queries
        @recipient_keyids = ($recipient);
    }
    else {
        dc_client_request_log($sid, 'RELAY works only for channels or key identifiers!', 'Request');
        return 1;
    }

    # add nickname key signature if nickname is set for channel
    push(@signature_keyids, $nickname_keyid) if ($nickname_keyid);

    # terminate data with a newline
    $data .= "\n" unless ($data =~ /.*\n$/);

    my $encrypted = crypt_base_data_encrypt($data, \@signature_keyids, \@recipient_keyids);
    my $request = 'RELAY' . ' ' . $recipient . "\n" . $encrypted;

    if (!$encrypted) {
        dc_client_request_log($sid, 'will not RELAY without encrypting request!', 'Request');
        return -1;
    }

    return dc_client_request($sid, $request);
}

sub dc_client_request_RELAY_MESSAGE($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel, $message) = @{$args};

    confess("dc_client_request_RELAY_MESSAGE(): no channel argument") unless($channel);
    confess("dc_client_request_RELAY_MESSAGE(): no message argument") unless($message);

    # terminate message with a newline
    $message .= "\n" unless ($message =~ /.*\n$/);

    my $message_request = 'MESSAGE' . ' ' . $channel . "\n" . $message;

    return dc_client_request_RELAY($sid, [ $channel, $message_request ]);
}

sub dc_client_request_RELAY_NICK($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel, $nickname_keyid) = @{$args};

    confess("dc_client_request_RELAY_NICK(): no channel argument") unless($channel);
    confess("dc_client_request_RELAY_NICK(): no nickname_keyid argument") unless($nickname_keyid);

    my $pubkey = crypt_base_key_export_pub($nickname_keyid);
    my $nick_request = 'NICK' . ' ' . $channel . "\n" . $pubkey;

    return dc_client_request_RELAY($sid, [ $channel, $nick_request ]);
}

sub dc_client_request_PING($$)
{
    my $sid = shift;
    my $args = shift;
    my $timestamp = strftime "%Y%m%d%H%M%S", localtime;

    my $request = 'PING' . ' ' . $timestamp;

    return dc_client_request($sid, $request);
}

sub dc_client_request_LIST($$)
{
    my $sid = shift;
    my $args = shift;
    my ($channel_pattern) = @{$args};

    my $request = 'LIST' . ' ' . $channel_pattern;

    return dc_client_request($sid, $request);
}

sub dc_client_request_REGISTER($$)
{
    my $sid = shift;
    my $args = shift;
    my ($name, $key_id, $role) = @{$args};

    my $pubkey = crypt_base_key_export_pub($key_id);
    my $request = 'REGISTER' . ' ' . uc($role) . ' ' . $name . "\n" . $pubkey;

    return dc_client_request($sid, $request);
}

sub dc_client_request_CHANNELSERVER($$)
{
    my $sid = shift;
    my $args = shift;
    my ($cmd) = @{$args};

    confess("dc_client_request_CHANNELSERVER(): no cmd argument") if (not $cmd);

    my $request = 'CHANNELSERVER' . ' ' . $cmd;

    return dc_client_request($sid, $request);
}

sub dc_client_request_log($$$)
{
    my ($sid, $message, $prefix) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $alias_customer = dc_session_data_get($sid, 'alias_customer');
    my $sap = dc_session_data_get($sid, 'sap');

    `echo "$prefix: $message" >> /tmp/request.log`;

    confess('no $alias_customer in dc_client_request_log()!') unless($alias_customer);

    # inform customer
    $poe_kernel->post($alias_customer, 'darkchannel_LOG_PROTO', $sid, $sap, $message, $prefix);
}

sub dc_client_request_send($$;$)
{
    my ($sid, $request, $request_args) = @_;

    return if ($request eq 'NOP');

    # call CLIENT REQUEST function (PRODUCER)
    my $request_func_name = 'dc_client_request_' . $request;
    dc_client_request_log($sid, "'" . $request_func_name . "()'", 'Request: Call Producer')
        if ($CONF->{log}->{log_dbg_transition});

    if ($request =~ /^[A-Z_]+$/) {
        # check if request function exists
        if (DarkChannel::Proto::Client::Request->can($request_func_name)) {
            # XXX: TODO: dont confess() but react accordingly on failed request generation
            my $prefix_content = 'Request';
            my $prefix_transport = 'Request Transport';
            my $request_func = \&$request_func_name;

            # call request function and store returned request data
            my $req = $request_func->($sid, $request_args)
                or confess("request generation failed in '$request_func_name'!");

            # if request function succeeded, send request
            if (ref($req) eq 'ARRAY') {
                my ($request_transport, $request_content) = @{ $req };

                # log request data if configured
                dc_client_request_log($sid, $request_content, $prefix_content)
                    if ($CONF->{log}->{log_dbg_request});
                dc_client_request_log($sid, $request_transport, $prefix_transport)
                    if ($CONF->{log}->{log_dbg_request_transport});

                # send request transport data  using an send event to channel server session
                $poe_kernel->post($sid, 'send', $request_transport);

                return 1;
            }

            return $req;
        }

        confess("request function '" . $request_func_name . "()' does not exist!")
    }

    confess("request command '" . $request . "' is not well formed [A-Z]!")
}

sub dc_client_request_initialize()
{
    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Client::Request");

    return 1;
}

1;
