#
# Dark Channel Client Protocol Interpreter Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Client::Interpreter;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;
use DarkChannel::Utils::ChannelStorage;

use DarkChannel::Proto::V1;
use DarkChannel::Proto::Client::Request;
use DarkChannel::Proto::Client::Response;

use DarkChannel::Node::Client::Conf;

# POE Debugging
#sub POE::Kernel::ASSERT_DEFAULT () { 1 }
#sub POE::Kernel::ASSERT_EVENTS  () { 1 }
#sub POE::Kernel::CATCH_EXCEPTIONS () { 0 }

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_client_interpreter_initialize
                  dc_client_interpreter_spawn );

sub i_log($$$)
{
    my ($sid, $message, $prefix) = @_;
    my $alias_customer = dc_session_data_get($sid, 'alias_customer');
    my $sap = dc_session_data_get($_[0], 'sap');

    # inform customer
    confess("i_log: sid is not a number!") unless ($_[0] =~ /^\d+$/);
    $poe_kernel->post($alias_customer, 'darkchannel_LOG_PROTO', $sid, $sap, $message, $prefix);

}

sub i_log_info($$;$)
{
    my ($sid, $message, $prefix) = @_;
    i_log($sid, $message, $prefix);
}

sub i_log_warn($$;$)
{
    my ($sid, $message, $prefix) = @_;
    $prefix = 'Warning' . ($prefix ? (': ' . $prefix) : '');
    i_log($sid, $message, $prefix);
}

sub i_log_err($$;$)
{
    my ($sid, $message, $prefix) = @_;
    $prefix = 'Error' . ($prefix ? (': ' . $prefix) : '');
    i_log($sid, $message, $prefix);
}

sub i_log_crit($$;$)
{
    my ($sid, $message, $prefix) = @_;
    $prefix = 'Critical' . ($prefix ? (': ' . $prefix) : '');
    i_log($sid, $message, $prefix);
}

sub i_log_dbg($$;$)
{
    my ($sid, $message, $prefix) = @_;
    i_log($sid, $message, $prefix) if ($CONF->{log}->{log_dbg_interpreter});
}

#
# join one or more new client(s) <cpubkey> to a channel <channel>
#
sub dc_client_interpreter_channelmember_join($$$)
{
    my $sid = shift;
    my $channel = shift;
    my $cpubkey = shift;

    my $now = strftime "%Y-%m-%d-%H%M", localtime;
    my $ckey = $sid . ':' . $channel;

    if (not dc_channel_exists($ckey)) {
        # create channel dataset
        my $cdata = {
            members => {},
        };

        # create new channel
        i_log_info($sid, "creating new channel '" . $channel . "'");
        i_log_info($sid, "joining new channel '" . $channel . "'", 'ChannelStorage');
        dc_channel_create($ckey, $cdata);

        # store channel name in channel server session
        dc_session_data_set($sid, 'channels', $channel, $now);
    }
    else {
        i_log_info($sid, "new client joining channel '" . $channel . "'");
    }

    # channel client data
    my $cdata = {
        time_join => $now,
        time_pong => time(),
        alias_channelserver => dc_session_data_get($sid, 'alias'),
        sap_direct => 0,
    };

    # register new clients on channel: import pubkeys in <cpubkey> and set channel data in ChannelStorage
    my $import = undef;
    if ($cpubkey)
    {
        # import new clients in $cpubkey
        $import = crypt_base_key_import($cpubkey);

        # store new clients as channel members in channel storage
        for my $ckeyid (keys %{ $import }) {
            i_log_info($sid, "added new client with key id ".$ckeyid." to channel '".$channel."'", 'ChannelStorage');
            dc_channel_data_set($ckey, 'members', $ckeyid, $cdata);
        }
    }

    return $import;
}

#
# part a channel member <ckeyid> from a joined channel <channel>
#
sub dc_client_interpreter_channelmember_part($$$)
{
    my $sid = shift;
    my $channel = shift;
    my $ckeyid = shift;

    my $ckey = $sid . ':' . $channel;

    # check if there is such a channel
    return undef unless (dc_channel_exists($ckey));

    # remove member from channel
    i_log_info($sid, "removed client with key id " . $ckeyid . " from channel '" . $channel . "' of session " . $sid, 'ChannelStorage');
    dc_channel_data_delete($ckey, 'members', $ckeyid);

    return $ckeyid;
}

#
#part a joined channel <channel>
#
sub dc_client_interpreter_channel_part($$)
{
    my $sid = shift;
    my $channel = shift;

    my $ckey = $sid . ':' . $channel;

    # check if there is such a channel
    return undef unless (dc_channel_exists($ckey));

    # leaving channel
    i_log_info($sid, "destroying channel '" . $channel . "'", 'ChannelStorage');
    my $ochannel = dc_channel_destroy($ckey);

    # remove channel name from channel server session
    dc_session_data_delete($sid, 'channels', $channel);

    return $ochannel;
}

#
# do protocol state transition and log accordingly
#
sub dc_client_interpreter_statetransition($$$)
{
    my $state_machine = shift;
    my $sid = shift;
    my $cmd = shift;

    my $protocol_version = $CONF->{protocol_version};
    my $product_name = $CONF->{product_name};
    my $proto_state = dc_session_data_get($sid, 'proto_state');
    my $alias = dc_session_data_get($sid, 'alias');

    # Fetch transition from transition matrix
    my ($request, $state) = dc_proto_transition($protocol_version, $state_machine, $proto_state, $cmd);

    my $response_request = '[' . $cmd . ' / ' . $request . ']';
    my $state_change  = $proto_state . '/' . $cmd . ' -> ' . $state . '/' . $request;
    i_log_dbg($sid, $state_change, 'Protocol: ' . $protocol_version)
        if ($CONF->{log}->{log_dbg_transition});

    # change current channel server's state
    i_log_dbg($sid, $state, 'Protocol: ' . $protocol_version . ': State')
        if ($CONF->{log}->{log_dbg_transition});
    dc_session_data_set($sid, 'proto_state', $state);

    return ($request, $state);
}

#
# spawn a new client interpreter
#
sub dc_client_interpreter_spawn($;$)
{
    my $state_machine = shift // $CONF->{service_name};
    my $alias_interpreter = shift // 'Client-Interpreter';

    my $debug = $CONF->{log}->{log_dbg_session_interpreter};
    my $session = POE::Session->create(

        options => { debug => $debug, trace => 0, default => 1 },

        inline_states => {
            _start => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # raport startup
                dc_log_dbg("client interpreter session created", $alias_interpreter);

                # set alias and store alias on heap
                $kernel->alias_set($alias_interpreter);
                $heap->{alias} = $alias_interpreter;

                # store state machine
                $heap->{state_machine} = $state_machine;
            },

            _stop => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
            },

            consume_WELCOME => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($pubkey, $pubkey_id) = @{$params};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "consuming 'WELCOME' event from sender (sid=" . $sid . ")", 'Protocol');

                # consume WELCOME: ignore the message's arguments, it's not signed. remember public key
                dc_session_data_set($sid, 'server', 'key_id', $pubkey_id) if ($pubkey_id);

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'WELCOME');

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_HELLO => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "consuming 'HELLO' event from sender (sid=" . $sid . ")", 'Protocol');

                # consume HELLO: store informations in session
                if ($arg =~ /^$CONF->{product_name}: ([A-Za-z ]+): (v[0-9]+): ([^\s]+)$/) {
                    my ($service_type, $proto_version, $node_type) = ($1, $2, $3);
                    dc_session_data_set($sid, 'server', 'protocol_version', $proto_version);
                    dc_session_data_set($sid, 'server', 'service_type', $service_type);
                    dc_session_data_set($sid, 'server', 'node_type', $node_type);
                }
                else {
                    confess("received 'HELLO' request but could not parse it's arguments (arg='" . $arg . "')!");
                }

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'HELLO');

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_JOIN => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $channel = $arg;
                my ($cpubkey) = @{$params};
                my $sm = $heap->{state_machine};
                my $channel_new = not dc_channel_exists($sid . ':' . $channel);
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $join_keyid = [];

                # consume JOIN: open notebook page, store channel information and import member public key
                i_log_dbg($sid, "consuming 'JOIN' event from sender (sid=".$sid.", channel=".$channel.")", 'Protocol');

                # add client to channel
                my $inspect = dc_client_interpreter_channelmember_join($sid, $channel, $cpubkey);

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'JOIN');

                # if channel is new add our own key_id
                push(@{ $join_keyid }, dc_session_data_get($sid, 'key_id')) if ($channel_new);

                # add key_id of all contained cpubkeys
                push(@{ $join_keyid }, $_) for (keys %{ $inspect });

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_JOIN', $sid, $channel, $join_keyid, $channel_new);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_PART => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($channel, $part_keyid) = split(' ', $arg);
                my $sm = $heap->{state_machine};
                my $channel_existed = dc_channel_exists($sid . ':' . $channel);
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $key_id = dc_session_data_get($sid, 'key_id');

                # consume PART: clear channel information and notify user of parting client
                i_log_dbg($sid, "consuming 'PART' event from sender (sid=". $sid . ", channel=" . $channel
                          . ", keyid=" . $part_keyid . ")", 'Protocol');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'PART');

                # check if client is leaving channel or if someone else is leaving the channel
                my $channel_parted = ($part_keyid eq $key_id) ? 1 : 0;

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_PART', $sid, $channel, $part_keyid, $channel_parted);

                # remove client from channel
                dc_client_interpreter_channelmember_part($sid, $channel, $part_keyid) unless ($channel_parted);
                dc_client_interpreter_channel_part($sid, $channel) if ($channel_parted);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_RELAY => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $recipient = $arg;
                my $sm = $heap->{state_machine};
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $key_id = dc_session_data_get($sid, 'key_id');

                # consume RELAY: next state
                i_log_dbg($sid, "consuming 'RELAY' event from sender (sid=" . $sid
                          . ", recipient=" . $recipient .")", 'Protocol');

                # check if recipient is a channel
                if ($recipient =~ /^#[a-zA-Z_-]+$/) {
                    # check if channel exists
                    my $channel_exists = dc_channel_exists($sid . ':' . $recipient);
                    unless($channel_exists) {
                        i_log_warn($sid, "received RELAY event for closed channel, ignoring message");
                        return;
                    }
                }
                else {
                    # check if recipient key identity is our client key identity
                    unless ($recipient eq $key_id) {
                        i_log_warn($sid, "received RELAY event for unkown client key (keyid=" . $recipient
                                   . ", ignoring message");
                        return;
                    }
                }

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'RELAY');

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_MESSAGE => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $recipient = $arg;
                my ($message, $client_keyid, $nickname_keyid) = @{$params};
                my $sm = $heap->{state_machine};
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $key_id = dc_session_data_get($sid, 'key_id');

                # consume MESSAGE: log it to notebook page
                i_log_dbg($sid, "consuming 'MESSAGE' event from sender (sid=" . $sid
                          . ", recipient=" . $recipient . ", client_keyid=" . $client_keyid
                          . ", nickname_keyid=" . $nickname_keyid . ")", 'Protocol');

                if ($recipient =~ /^#[a-zA-Z_-]+$/) {
                    # check if channel exists
                    my $channel_exists = dc_channel_exists($sid . ':' . $recipient);
                    unless($channel_exists) {
                        i_log_warn($sid, "received MESSAGE event for closed channel, ignoring message");
                        return;
                    }

                    # log message
                    i_log_dbg($sid, $message, 'Channel ' . $recipient);
                }
                else {
                    # check if recipient key identity is our client key identity
                    unless ($recipient eq $key_id) {
                        i_log_warn($sid, "received MESSAGE event for unkown client key (keyid=" . $recipient
                                   . ", ignoring message");
                        return;
                    }

                    # log message
                    i_log_dbg($sid, $message, 'Private Message');
                }

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'MESSAGE');

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_MESSAGE', $sid, $recipient,
                              $client_keyid, $nickname_keyid, $message);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_NICK => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my $channel = $arg;
                my ($nick_pubkey, $nick_keyid, $nick_fq, $client_keyid) = @{$params};
                my $sm = $heap->{state_machine};
                my $channel_exists = dc_channel_exists($sid . ':' . $channel);
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');

                # consume NICK: import pubkey to recognize next msg signature and inform user of nickname change
                i_log_dbg($sid, "consuming 'NICK' event from sender (sid=" . $sid . ", channel=" . $channel
                          . ", nick_keyid=" . $nick_keyid . ", client_keyid=" . $client_keyid . ")",
                          'Protocol');

                # check if channel exists
                unless($channel_exists) {
                    i_log_warn($sid, "received NICK event for closed channel, ignoring message");
                    return;
                }

                # import nick public key material
                my $import = crypt_base_key_import($nick_pubkey);

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'NICK');

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_NICK', $sid, $channel, $client_keyid, $nick_keyid, $nick_fq);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_PONG => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($timestamp) = $arg;
                my $sm = $heap->{state_machine};
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $key_id = dc_session_data_get($sid, 'key_id');
                my $now = strftime "%Y%m%d%H%M%S", localtime;
                my $lag = ($now - $timestamp);

                # consume PONG: log ping time
                i_log_dbg($sid, "consuming 'PONG' event from sender (sid=". $sid . ", timestamp=" . $timestamp
                          . ", lag=" . $lag . ")", 'Protocol');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'PONG');

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_PONG', $sid, $timestamp, $lag);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_LIST => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($list) = @{$params};
                my $sm = $heap->{state_machine};
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                my $key_id = dc_session_data_get($sid, 'key_id');

                # consume LIST: log channels
                i_log_dbg($sid, "consuming 'LIST' event from sender (sid=". $sid . ")", 'Protocol');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'LIST');

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_LIST', $sid, $list);

                # send request
                dc_client_request_send($sid, $request);
            },

            consume_REGISTER => sub {
                my ($kernel, $heap, $session, $sid, $arg, $params)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];
                my ($role, $name, $pubkey, $pubkey_id) = @{$params};
                my $sm = $heap->{state_machine};
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');

                # consume REGISTER: import signed public key, inform customer
                i_log_dbg($sid, "consuming 'REGISTER " . uc($role) . "' event from sender (sid=" . $sid
                          . ", role=" . $role . ", name=" . $name . ", keyid=" . ($pubkey_id // '0') .")",
                          'Protocol');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'REGISTER');

                # inform customer
                $kernel->post($alias_customer, 'darkchannel_REGISTER',
                              $sid, $role, $name, $pubkey_id, $pubkey);

                # send request
                dc_client_request_send($sid, $request);
            },

            cmd_JOIN => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($channel) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "JOIN command event for channel " . $channel . " (sid=" . $sid . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_JOIN');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_PART => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($channel) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "PART command event for channel " . $channel . " (sid=" . $sid . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_PART');

                # part the channel
                i_log_info($sid, "leaving channel '" . $channel . "'");
                dc_client_interpreter_channel_part($sid, $channel);

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_RELAY_MESSAGE => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($channel, $message) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "RELAY_MESSAGE command event for channel " . $channel . " (sid=" . $sid . ")",
                          'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_RELAY_MESSAGE');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_RELAY_NICK => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($channel, $nickname_keyid, $nickname) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "RELAY_NICK command event for channel " . $channel . " (sid=" . $sid
                          . ", nickname=" . $nickname . ", nickname_keyid=" . $nickname_keyid . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_RELAY_NICK');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_PING => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "PING command event (sid=" . $sid . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_PING');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_LIST => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($channel_pattern) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "LIST command event (sid=" . $sid . ", channel_pattern=" . $channel_pattern
                          . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_LIST');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_REGISTER => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($name, $key_id, $role) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "REGISTER command event (sid=" . $sid . ", name=" . $name . ", keyid=" . $key_id
                          . ", role=" . $role . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_REGISTER');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            cmd_CHANNELSERVER => sub {
                my ($kernel, $heap, $session, $sid, $args)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($cmd) = @{$args};
                my $sm = $heap->{state_machine};

                i_log_dbg($sid, "CHANNELSERVER command event for command " . $cmd . " (sid=" . $sid . ")", 'Command');

                # protocol state transition (-> next-state, response)
                my ($request, $state) = dc_client_interpreter_statetransition($sm, $sid, 'cmd_CHANNELSERVER');

                # send request
                dc_client_request_send($sid, $request, $args);
            },

            process_response => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $response) = @_[ARG0, ARG1];

                my $success = dc_client_response_process($sid, $response);
            },

            client_connected => sub {
                my ($kernel, $heap, $session, $sender) = @_[KERNEL, HEAP, SESSION, SENDER];
                my ($alias, $sap, $sid, $key_id, $alias_customer, $opaque_sdata)
                    = @_[ARG0, ARG1, ARG2, ARG3, ARG4, ARG5];

                # channel server session data
                my $now = strftime "%Y-%m-%d-%H%M", localtime;
                my $opaque = $opaque_sdata ? $opaque_sdata : {}; # make sure we have a hash of opaque session data
                my $sdata = {
                    %{ $opaque }, # add opaque session data

                    sap => $sap,
                    alias => $alias,
                    key_id => $key_id,
                    alias_customer => $alias_customer,
                    time_connect => $now,
                    proto_state => 'initialization',
                };

                # register channel server's session
                dc_session_register($sid, $sdata);

                # inform consumer of new client, call it synchroniously making sure everything
                # is setup (eg. notebook page open) when the first events arrive (eg. LOG_xxx)
                $kernel->call($alias_customer, 'darkchannel_CONNECTED', $sid);

                # new channel server connection
                i_log_info($sid, "channel server connected (sid=" . $sid . ", keyid=". $key_id . ")");
                i_log_dbg($sid, "configuring interpreter to inform customer '" . $alias_customer. "'");
            },

            client_disconnected => sub {
                my ($kernel, $heap, $session, $alias, $sap, $sid)
                    = @_[KERNEL, HEAP, SESSION, ARG0, ARG1, ARG2];

                # lost channel server connection
                i_log_info($sid, "channel server disconnected (sid=" . $sid . ")");

                # channel data update: destroy all channels for this session
                for (dc_channel_find('^' . $sid . ':')) {
                    dc_client_interpreter_channel_part($sid, $1) if (/$sid:(.+)/);
                }

                # close notebook page
                my $alias_customer = dc_session_data_get($sid, 'alias_customer');
                $kernel->call($alias_customer, 'darkchannel_DISCONNECTED', $sid);

                # unregister channel server's session if still existing. forced client shutdown
                # leads to mass disconnecting/unregistering which can remove a session prior to
                # this event
                dc_session_unregister($sid) if (dc_session_exists($sid));
            },
        },
    );
}

sub dc_client_interpreter_initialize()
{
    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Client::Interpreter");

    return 1;
}

1;
