#
# Dark Channel Terminal Notebook Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Term::Notebook;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Curses;
use Curses::UI::Common;
use Curses::UI::Widget;

use base qw(Curses::UI::Notebook);

#
# Curses::UI::Notebook has several bugs which, depending on the bug have or have not
# been fixed in the latest release of Curses::UI. To make sure everybody has a fixed
# version of Curses::UI::Notebook, this class subclasses Curses::UI::Notebook and
# overrides the found and partly fixed bugs with a correct implementation of the
# delete_page() function. This assures everybody (irrelevant what version they have)
# will use a fixed function.
#

#
# delete a notebook page
#
sub delete_page($$)
{
    my $this = shift;
    my $page = shift or return;

    # Make sure page is part of the notebook.
    $this->root->fatalerror("The notebook widget does not have a page named '$page'!")
        unless (defined $this->{-id2object}->{$page});

    Curses::UI::Notebook->debug_msg("  deleting '$page' page");

    if (defined $this->{callback}{$page}{-on_delete}) {
        Curses::UI::Notebook::debug_msg("  calling delete callback for $page");
        $this->{callback}{$page}{-on_delete}->($this,$page);
    }

    my $active_page = $this->active_page;
    @{$this->{-pages}} = grep($page ne $_, @{$this->{-pages}});
    $this->activate_page($this->first_page) if ($page eq $active_page);

    # HOTFIX: do not call SUPER::DESTROY or the whole notebook will get destroyed
    #$this->SUPER::DESTROY($page);

    # HOTFIX: Curses::UI::Notebook needs to delete the -id2object entry or it will
    #         fail to create a new page with the same name as the old one
    delete $this->{-id2object}->{$page};

    return;
}

#
# find notebook page by regex
#
sub find_pages($$)
{
    my $this = shift;
    my $pattern = shift or confess("Notebook->find_pages() without pattern called!");

    Curses::UI::Notebook->debug_msg("  find page(" . $pattern . ")");

    my @found = grep(/$pattern/, @{$this->{-pages}});
    return @found;
}

#
# return the content object of a page (eg. the TextEditor inside the Notebook page)
#
sub active_page_obj($)
{
    my $this = shift;
    my $page = $this->active_page();
    my $winobj = $this->getobj($page);
    my $pageobj = $winobj->getobj('text-' . $page);

    return $pageobj;
}

#
# scroll down page-wise of active notebook page
#
sub active_pagedown()
{
    my $this = shift;
    my $obj = $this->active_page_obj();
    $obj->cursor_pagedown();
    $obj->draw();
}

#
# scroll up page-wise of active notebook page
#
sub active_pageup()
{
    my $this = shift;
    my $obj = $this->active_page_obj();
    $obj->cursor_pageup();
    $obj->draw();
}

#
# scroll down line-wise of active notebook page
#
sub active_linedown()
{
    my $this = shift;
    my $obj = $this->active_page_obj();
    $obj->cursor_down();
    $obj->draw();
}

#
# scroll up line-wise of active notebook page
#
sub active_lineup()
{
    my $this = shift;
    my $obj = $this->active_page_obj();
    $obj->cursor_up();
    $obj->draw();
}

1;
