#
# Dark Channel IRC Core Listener Client Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Core::IRCListener;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;
use DarkChannel::Utils::ChannelStorage;

use DarkChannel::Proto::Client;

use DarkChannel::Node::Client::Core::IRC;
use DarkChannel::Node::Client::Core::Log;
use DarkChannel::Node::Client::Conf;

# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Component::Server::IRC);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_poe_core_ircd_initialize
                  dc_poe_core_ircd_spawn );

my $alias_interpreter = 'Client-Interpreter';
my $alias_terminal = 'Client-Terminal';

sub dc_poe_core_ircd_check_password($$$)
{
    my ($name, $pass, $alias) = @_;

    confess('no $name or no $pass in dc_poe_core_ircd_check_password()') unless($name && $pass && $alias);

    # fetch stored hashed passphrase
    my $conf = $CONF->{node}->{channelserver}->{$name};
    my $stored_pass = $conf->{passphrase};

    # hash given password $pass
    my $hashed_pass = crypt_base_data_bcrypt($pass, $stored_pass);

    # check authorization
    my $authorized = 0;
    $authorized = 1 if ($stored_pass && $hashed_pass && ($hashed_pass eq $stored_pass));

    core_log_info("checking core user password (name=" . $name . "): authorized: " . ($authorized ? "YES" : "NO"), $alias);
    return $authorized;
}


sub dc_poe_core_ircd_set_password($$$)
{
    my ($name, $pass, $alias) = @_;

    confess('no $name or no $pass in dc_poe_core_ircd_set_password()') unless($name && $pass && $alias);

    core_log_info("setting new password for core user (name=" . $name . ")", $alias);

    # hash given password $pass
    my $hashed_pass = crypt_base_data_bcrypt($pass, undef);

    # store hashed password
    $CONF->{node}->{channelserver}->{$name}->{passphrase} = $hashed_pass;       # make sure we remember the pass

    return $hashed_pass;
}

sub dc_poe_core_ircd_spawn(;$)
{
    my $alias_customer = shift // '';
    my $core_conf = ($CONF->{node}->{core}->{ircd}->{_default}) // 0;

    die("dc_poe_core_ircd_spawn() no CONF->{node}->{core}->{ircd}->{_default} found!")
        unless($core_conf);

    my $bind_host = ($core_conf->{bind_host}) // die("dc_poe_core_ircd_spawn() no _default host found in CONF!");
    my $bind_port = ($core_conf->{bind_port}) // die("dc_poe_core_ircd_spawn() no _default port found in CONF!");

    my $sap = $bind_host . ':' . $bind_port;
    my $alias_prefix = 'Core-';
    my $alias_listener = $alias_prefix . 'IRC-Listener-' . $sap;
    my $alias_irc_interpreter = $alias_prefix . 'IRC-Interpreter-' . $sap;
    my $debug = ($CONF->{log}->{log_dbg_session_core_ircd}) // 0;

    my $admin = [ split(/|/, $core_conf->{server_admin}) ];
    my $info = [ split(/|/, $core_conf->{server_info}) ];

    # IRCD configuration
    my $ircd_conf = {
        servername => $core_conf->{server_name},
        serverdesc => $core_conf->{server_description},
        network    => $core_conf->{server_network},
        admin      => $admin,
        info       => $info,
        nicklen    => 20,
        userlen    => 20,
        auth       => 0,
    };

    # SSL configuration
    my $sslify = {
        sslify_options => [
            $core_conf->{server_ssl_key},
            $core_conf->{server_ssl_crt},
            $core_conf->{server_ssl_ver},
            $core_conf->{server_ssl_opt}
        ],
    };

    # check SSL configuration
    my $err = '';
    ($err, $sslify) = ("no 'server_ssl_key' in dcc.conf!", {}) unless($core_conf->{server_ssl_key});
    ($err, $sslify) = ("no 'server_ssl_key' in dcc.conf!", {}) unless($core_conf->{server_ssl_crt});
    ($err, $sslify) = ("could not read ssl key file " . $core_conf->{server_ssl_key} . "!", {})
        unless (-r $core_conf->{server_ssl_key});
    ($err, $sslify) = ("could not read ssl certificate file " . $core_conf->{server_ssl_crt} . "!", {})
        unless (-r $core_conf->{server_ssl_crt});

    # don't start IRC Listener and fail with an error message if there was an error
    if ($err) {
        dc_log_crit("ATTENTION: could not activate SSL! check your configuration settings!\n" .
                   "ERROR: " .$err . "\n" .
                   "disabling IRC core support...", $alias_listener);
        return (-1, $alias_listener);
    }

    # create an instance of our own implementation of POE::Component::Server::IRC and pass it to the IRC Listener
    my $core_ircd = DarkChannel::Node::Client::Core::IRC->spawn(
        config => $ircd_conf,
        debug => $debug,
        alias => $alias_irc_interpreter,
        alias_customer => $alias_customer,

        # this needs POE::Component::SSLify
        %{ $sslify },
    );

    # create IRC Listener session
    my $session_id = POE::Session->create(
        options => { debug => $debug, trace => 0, default => 1 },

        inline_states => {
            _start => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                my $usessl = $sslify ? 1 : 0;

                # set alias
                $kernel->alias_set($alias_listener);
                $heap->{alias} = $alias_listener;
                $heap->{alias_customer} = $alias_customer;

                # register ircd events
                $heap->{ircd}->yield('register', 'all');

                # log bind
                core_log_dbg("binding to " . $bind_host . ":" . $bind_port . " (usessl=" . $usessl . ")", $alias_listener);
                core_log_dbg("using SSL key " . $core_conf->{server_ssl_key}, $alias_listener) if ($usessl);
                core_log_dbg("using SSL certificate " . $core_conf->{server_ssl_crt}, $alias_listener) if ($usessl);

                # start a listener on the 'standard' IRC port.
                $heap->{ircd}->add_listener(bindaddr => $bind_host, port => $bind_port, usessl => $usessl);
            },

            _default => sub {
                my ($event, $args) = @_[ARG0 .. $#_];
                my $session = $poe_kernel->alias_resolve(($poe_kernel->alias_list())[0]);
                my $sid = $session->ID;
                my $alias = dc_session_data_get($sid, 'alias');

                # only look at ircd_core_* events
#                if ($event !~ /^ircd_core_/) {
#                    # and silently ignore ircd_cmd_* events
#                    if ($event !~ /^ircd_cmd_/) {
#                        core_log_dbg("event not of type 'ircd_[core|cmd]_*', ignoring" , $alias . ': Event: ' . $event);
#                    }
#                    return;
#                }

                dc_log_dbg("ignoring event:\n" . Dumper($args), $alias . ': Event: ' . $event);
            },

            _child => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($event, $args) = @_[ARG0 .. $#_];
                my $sid = $session->ID;
                my $alias = dc_session_data_get($sid, 'alias');

                dc_log_dbg("ignoring event", $alias . ": child event '" . $event . "'");
            },

            shutdown => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];

                $heap->{ircd}->shutdown();
            },

            # prepare connect to channel server
            core_connect_channelserver_prepare => sub {
                my ($kernel, $session, $heap, $csession) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $name, $nick, $key_id, $key_pass) = @_[ARG0, ARG1, ARG2, ARG3, ARG4, ARG5];

                # check if cryptosystems need to generate key material
                my $no_delay = crypt_base_check_cryptosystems($key_id);

                # update state: key material generation
                if ($no_delay) {
                    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'key_material_found');
                }
                else {
                    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'key_material_generation');
                }

                # if there will be a delay, notify user
                if ($no_delay) {
                    $heap->{ircd}->send_notice($sid, $conn_id, "Key material for " . $nick . " found");
                    core_log_info("core found needed key material for (sid=" . $sid . ", conn_id=" . $conn_id
                                . ", nick=" . $nick . ", name=" . $name . ")", $alias_listener);
                }
                else {
                    my @msg_delay = (
                        "Generating new key material",
                        "This process can take up to 5 minutes,",
                        "without an entropy gathering daemon.",
                        "Please wait..."
                    );
                    $heap->{ircd}->send_notice($sid, $conn_id, $_) for (@msg_delay);
                    core_log_info("core creates key material for (sid=" . $sid . ", conn_id=" . $conn_id
                                . ", nick=" . $nick . ", name=" . $name . ")", $alias_listener);
                }

                $kernel->delay('core_connect_channelserver', 1, $sid, $conn_id, $name, $nick, $key_id, $key_pass, not $no_delay);
            },

            core_connect_channelserver => sub {
                my ($kernel, $session, $heap, $csession) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $name, $nick, $key_id, $key_pass, $delay) = @_[ARG0, ARG1, ARG2, ARG3, ARG4, ARG5, ARG6];

                # fetch channel server host/port
                my $host = $CONF->{node}->{channelserver}->{$name}->{host};
                my $port = $CONF->{node}->{channelserver}->{$name}->{port};
                my $sap = $host . ':' . $port;

                # setup cryptosystems, we will use the resulting keyid as our identity
                $key_id = crypt_base_setup_cryptosystems($host, $key_id, $key_pass);

                if ($key_id) {
                    if($delay) {
                        # inform client: key material found
                        $heap->{ircd}->send_notice($sid, $conn_id, "Key generation successful!");
                        core_log_info("core successfully created key material for (sid=" . $sid . ", conn_id=" . $conn_id
                                    . ", nick=" . $nick . ", name=" . $name . ")", $alias_listener);
                    }

                    # inform client: show key information
                    my $info = crypt_base_key_information_str($key_id);
                    my @info = split(/\n/, $info);
                    #$heap->{ircd}->send_notice($sid, $conn_id, "Using key id " . $key_id);
                    $heap->{ircd}->send_notice($sid, $conn_id, $_) for (@info);

                    # store channel server's keyid in $name
                    $CONF->{node}->{channelserver}->{$name}->{key_id} = $key_id;

                    # update state: connecting
                    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'connecting');

                    # inform client: connecting to channel server
                    $heap->{ircd}->send_notice($sid, $conn_id, "Connecting to channel server at " . $sap . " ...");
                    core_log_info("core connecting to channel server (sid=" . $sid . ", conn_id=" . $conn_id
                                . ", nick=" . $nick . ", name=" . $name . ", key_id=" . $key_id . ", sap=" . $sap
                                . ")", $alias_listener);

                    # create new DarkChannel::Node::Client::ChannelServer client if key material is ok
                    my $alias_prefix_channelserver = $alias_prefix . $nick . '-' . $conn_id . '-';
                    my $opaque_sdata = {
                        core => {
                            sid => $sid,
                            conn_id => $conn_id,
                        },
                    };
                    my ($sid_channelserver, $alias) = dc_poe_client_spawn($host, $port, $key_id,
                                                                                 $alias_listener,
                                                                                 $alias_prefix_channelserver,
                                                                                 $opaque_sdata);

                    # store channel server client information in core client state
                    my $cdata = {
                        sid => $sid_channelserver,
                        sap => $sap,
                        key_id => $key_id,
                    };
                    dc_session_data_set($sid, 'clients', $conn_id, 'channelserver', $cdata);
                }
                else {
                    confess("failed to setup DarkChannel::Crypt::Base crypto systems!");

                    # update state: key material failed
                    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'key_material_failed');

                    # disconnect client: failed to generate key material
                    core_log_crit("core failed to create key material for (sid=" . $sid . ", conn_id=" . $conn_id
                                . ", nick=" . $nick . ", name=" . $name . ")", $alias_listener);
                    my $reason = [
                        "Failed to generate key material!",
                        "Please contact an administrator, this service is unavailable...",
                    ];
                    $kernel->yield('core_client_disconnect', $sid, $conn_id, $reason);
                }

                return;
            },

            # disconnect client asap and notify client
            core_client_disconnect => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $reason) = @_[ARG0, ARG1, ARG2];

                $reason = [ $reason ] unless(ref($reason) eq 'ARRAY');
                $heap->{ircd}->send_notice($sid, $conn_id, $_) for (@{$reason});

                # terminate client delayed to make sure notices get sent out first
                $kernel->delay('core_client_terminate', 1, $sid, $conn_id, $reason->[0]);
            },

            # terminate client immediately
            core_client_terminate => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $reason) = @_[ARG0, ARG1, ARG2];

                # shutdown the channelserver if there is a session
                if (my $c = dc_session_data_get($sid, 'clients', $conn_id)) {
                    my $sid_channelserver = $c->{channelserver}->{sid};
                    $kernel->post($sid_channelserver, 'shutdown') if ($sid_channelserver);
                }

                # terminate connection with error message
                $heap->{ircd}->terminate_connection($sid, $conn_id, $reason);
            },

            ircd_registered => sub {
                my ($kernel, $session, $heap, $csession) = @_[KERNEL, SESSION, HEAP, ARG0];
                my $sid = $session->ID;

                # new ircd core listener
                core_log_info("core listening (sid=" . $sid . ", protocol=irc)", $alias_listener);

                # channel server session data
                my $now = strftime "%Y-%m-%d-%H%M", localtime;
                my $sdata = {
                    sap => $sap,
                    alias => $alias_listener,
                    time_connect => $now,
                };

                # register channel server's session
                dc_session_register($sid, $sdata);
            },

            # client connected
            ircd_core_client_connected => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id) = @_[ARG0, ARG1, ARG2];

                core_log_info("core client connected (sid=" . $sid . ", conn_id=" . $conn_id . ", protocol=irc)",
                            $alias_listener);
            },

            # client identified
            ircd_core_client_identified => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $nick) = @_[ARG0, ARG1, ARG2];
                my $cap = dc_session_data_get($sid, 'clients', $conn_id, 'state_ircd_cap') // 'disabled';
                my $name = '_core_' . $nick;

                if ($cap eq 'disabled') {
                    # check if the given nick already has a channel server entry
                    my $conf = $CONF->{node}->{channelserver}->{$name};

                    if ($conf) {
                        # inform client: this is a registered nick
                        $heap->{ircd}->send_notice($sid, $conn_id, "ATTENTION: Nick " . $nick . " is a registered name!");
                        $heap->{ircd}->send_notice($sid, $conn_id, "Please authenticate using '/pass <phrase>'");
                        $heap->{ircd}->send_notice($sid, $conn_id, "or by using SASL authentication while logging in");
                        $heap->{ircd}->send_notice($sid, $conn_id, "You can change your nick name with '/nick <name>'");
                    }
                    else {
                        # inform client: this is not a registered nick, you can register it
                        $heap->{ircd}->send_notice($sid, $conn_id, "The account '" . $nick . "' is not registered");
                        $heap->{ircd}->send_notice($sid, $conn_id, "Register a new account with /pass <phrase>");
                        $heap->{ircd}->send_notice($sid, $conn_id, "or change your nick with /nick <name>");
                        $heap->{ircd}->send_notice($sid, $conn_id, "After registration you can authenticate using SASL");
                        $heap->{ircd}->send_notice($sid, $conn_id, "during login or by authenticating with /pass again");
                    }
                }
            },

            # client auth received
            ircd_core_client_auth_done => sub {
                my ($kernel, $session, $heap, $csession) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $nick, $pass) = @_[ARG0, ARG1, ARG2, ARG3];
                my $alias = dc_session_data_get($sid, 'alias');
                my $sap = dc_session_data_get($sid, 'clients', $conn_id, 'sap_peer');
                my $name = '_core_' . $nick;

                # client auth done, check nick & pass
                core_log_info("core authentication received (sid=" . $sid . ", conn_id=" . $conn_id
                            . ", nick=" . $nick . ", pass=***)", $alias_listener);

                # inform client: authentication received
                $heap->{ircd}->send_notice($sid, $conn_id, "Authentication received");

                # check if the given nick is a valid nick name
                if ($nick =~ /^0x/) {
                    # nick names starting with 0x are not valid
                    core_log_err("illigal nick name '" . $nick . "' used, "
                                 . "terminating connection from " . $sap , $alias_listener);

                    my $reason = [
                        "Nick name rejected!",
                        "Choose a different nick and reconnect!",
                        ];
                    $kernel->yield('core_client_disconnect', $sid, $conn_id, $reason);
                }

                # check if the given nick already has a channel server entry
                my $conf = $CONF->{node}->{channelserver}->{$name};

                if ($conf) {
                    my $host = $conf->{host};
                    my $port = $conf->{port};
                    my $key_id = $conf->{key_id};
                    my $passphrase = $conf->{passphrase};

                    if ($host && $port && $key_id && $passphrase) {
                        if (dc_poe_core_ircd_check_password($name, $pass, $alias)) {
                            # inform client: authentication accepted
                            $heap->{ircd}->send_notice($sid, $conn_id, "Authentication accepted");

                            # user nick identified as a registered user, connect to channel server
                            # XXX: TODO: allow use of /connect <cs> command to connect if automatic_connect=false
                            $kernel->yield('core_connect_channelserver_prepare', $sid, $conn_id, $name, $nick, $key_id, $pass);
                        }
                        else {
                            # could not identify nick name, given passphrase is wrong
                            core_log_err("core authentication failed for registered nick '" . $nick
                                       . "' failed, terminating connection from " . $sap , $alias_listener);

                            my $reason = [
                                "Authentication rejected!",
                                "Choose a different nick and reconnect!",
                            ];
                            $kernel->yield('core_client_disconnect', $sid, $conn_id, $reason);
                        }
                    }
                    else {
                        # inform client: no valid configuration found
                        $heap->{ircd}->send_notice($sid, $conn_id, "Identity record for " . $nick . " is invalid");
                        $conf = undef;
                    }
                }
                unless ($conf) {
                    # given nick has no registered entry,

                    # inform client: this is a new nick, we will register it
                    $heap->{ircd}->send_notice($sid, $conn_id, "Registering '" . $nick . "' with given password");

                    # fetch default host/port for later usage
                    my $default_host = $CONF->{node}->{channelserver}->{_default}->{host};
                    my $default_port = $CONF->{node}->{channelserver}->{_default}->{port};
                    my $default_key_id = 0;

                    # hash password
                    dc_poe_core_ircd_set_password($name, $pass, $alias);

                    # store host/port/key_id for later usage in '$nick' channel server entry
                    $CONF->{node}->{channelserver}->{$name}->{host} = $default_host;
                    $CONF->{node}->{channelserver}->{$name}->{port} = $default_port;
                    $CONF->{node}->{channelserver}->{$name}->{key_id} = $default_key_id; # make sure we get a new key

                    # user nick identified as a new user, connect to channel server
                    $kernel->yield('core_connect_channelserver_prepare', $sid, $conn_id, $name, $nick, $default_key_id, $pass)
                }
            },

            ircd_core_client_cmd_join => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $channel) = @_[ARG0, ARG1, ARG2];
                my $alias = dc_session_data_get($sid, 'alias');
                my $recipient_sid = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'sid');

                core_log_dbg("core JOIN command from client (sid=" . $sid . ", recipient_sid=" . $recipient_sid
                           . ", channel=" . $channel . ")", $alias);
                # make interpreter execute a JOIN request to $recipient
                $kernel->post($alias_interpreter, 'cmd_JOIN', $recipient_sid, [ $channel ]) if ($channel);
            },

            ircd_core_client_cmd_part => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $channel) = @_[ARG0, ARG1, ARG2];
                my $alias = dc_session_data_get($sid, 'alias');
                my $recipient_sid = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'sid');

                core_log_dbg("core PART command from client (sid=" . $sid . ", recipient_sid=" . $recipient_sid
                           . ", channel=" . $channel . ")", $alias);
                # make interpreter execute a PART request to $recipient
                $kernel->post($alias_interpreter, 'cmd_PART', $recipient_sid, [ $channel ]) if ($channel);
            },

            ircd_core_client_cmd_privmsg => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $channel, $message) = @_[ARG0, ARG1, ARG2, ARG3];
                my $alias = dc_session_data_get($sid, 'alias');
                my $recipient_sid = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'sid');

                core_log_dbg("core PRIVMSG command from client (sid=" . $sid . ", recipient_sid=" . $recipient_sid
                           . ", channel=" . $channel . "): " . $message, $alias);
                # make interpreter execute a RELAY request to $recipient
                $kernel->post($alias_interpreter, 'cmd_RELAY', $recipient_sid, [ $channel, $message ])
                    if ($recipient_sid && $channel && $message);
            },

            ircd_core_client_cmd_names => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $channels) = @_[ARG0, ARG1, ARG2];
                my $alias = dc_session_data_get($sid, 'alias');
                my $sid_channelserver = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'sid');
                my $key_id = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'key_id');

                # go through all the given channels and reply with all member key_id
                core_log_dbg("core NAMES command from client (sid=" . $sid . ", key_id=" . $key_id . ", channels="
                           . join(", ", @{ $channels }) . ")", $alias);
                my $names = {};
                foreach my $channel (@{ $channels }) {
                    my $ckey = $sid_channelserver . ':' . $channel;
                    my $members = dc_channel_data_get($ckey, 'members');
                    my @ids = ();
                    push(@ids, keys %{ $members }); # all all channel members
                    push(@ids, $key_id);            # add ourselfes, we are on the channel too
                    $names->{$channel} = \@ids;
                }

                $heap->{ircd}->send_rpl_names($sid, $conn_id, $names);
            },

            ircd_core_client_cmd_quit => sub {
                my ($kernel, $session, $heap) = @_[KERNEL, SESSION, HEAP];
                my ($sid, $conn_id, $channels) = @_[ARG0, ARG1, ARG2];
                my $alias = dc_session_data_get($sid, 'alias');
                my $sid_channelserver = dc_session_data_get($sid, 'clients', $conn_id, 'channelserver', 'sid');

                # go through all the given channels and reply with all member key_id
                core_log_dbg("core QUIT command from client (sid=" . $sid . ")", $alias);

                # shutdown the channelserver
                $kernel->post($sid_channelserver, 'shutdown' );
            },

            darkchannel_CONNECTED => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $sid_channelserver = $_[ARG0];
                my $sid = $session->ID;

                core_log_dbg("darkchannel_CONNECTED (sid=" . $sid . ", sid_channelserver=" . $sid_channelserver . ")",
                           $alias_listener);

                # find core sid/conn_id/key_id
                my $core_sid = dc_session_data_get($sid_channelserver, 'core', 'sid');
                my $core_conn_id = dc_session_data_get($sid_channelserver, 'core', 'conn_id');
                my $key_id = dc_session_data_get($sid_channelserver, 'key_id');
                my $sap = dc_session_data_get($sid_channelserver, 'sap');

                # update client session state
                if ($core_conn_id) {
                    # update state: connected
                    dc_session_data_set($core_sid, 'clients', $core_conn_id, 'state_ircd', 'connected');

                    # inform client: connected
                    $heap->{ircd}->send_notice($core_sid, $core_conn_id, "Connected to " . $sap);

                    # change client nickname to key_id
                    $heap->{ircd}->send_nick($core_sid, $core_conn_id, $key_id);

                    # send delayed userhost using keyid (we did not send a RPL_USERHOST when requested until 'connected'
                    $heap->{ircd}->send_rpl_userhost($core_sid, $core_conn_id, [$key_id]);
                }
                else {
                    core_log_err("darkchannel_CONNECTED (sid_channelserver=" . $sid_channelserver
                               . "): could not find client session for channel server!", $alias_listener);
                }
            },

            darkchannel_DISCONNECTED => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $sid_channelserver = $_[ARG0];
                my $sid = $session->ID;

                core_log_dbg("darkchannel_DISCONNECTED (sid=" . $sid . ", sid_channelserver=" . $sid_channelserver . ")",
                           $alias_listener);

                # find core sid/conn_id
                my $core_sid = dc_session_data_get($sid_channelserver, 'core', 'sid');
                my $core_conn_id = dc_session_data_get($sid_channelserver, 'core', 'conn_id');

                # update client session state
                if ($core_conn_id) {
                    if (dc_session_data_get($core_sid, 'clients', $core_conn_id)) {
                        # update state: disconnected
                        dc_session_data_set($core_sid, 'clients', $core_conn_id, 'state_ircd', 'disconnected');

                        # inform client and disconnect
                        my $reason = [
                            "Channel server closed connection!",
                            "Please reconnect...",
                            ];
                        $kernel->yield('core_client_disconnect', $core_sid, $core_conn_id, $reason);
                    }
                }
                else {
                    core_log_err("darkchannel_DISCONNECTED (sid_channelserver=" . $sid_channelserver
                               . "): could not find client session for channel server!", $alias_listener);
                }
            },

            darkchannel_LOG_PROTO => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $sap, $message, $prefix) = @_[ARG0, ARG1, ARG2, ARG3];

                my $cs = dc_session_data_get($sid);
                return unless($cs);
                my $core_sid =$cs->{core}->{sid};
                my $core_conn_id = $cs->{core}->{conn_id};
                my $debug = $CONF->{log}->{log_dbg_session_core_irc_proto};
                $prefix = $prefix // '';

                # send NOTICE to client
                $heap->{ircd}->send_notice($core_sid, $core_conn_id, $message, $prefix) if ($debug);

                # log to core log
                core_log_dbg($message, $prefix);
            },

            darkchannel_LOG_CHANNELSERVER => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $sap, $message, $prefix) = @_[ARG0, ARG1, ARG2, ARG3];

                my $cs = dc_session_data_get($sid);
                return unless($cs);
                my $core_sid =$cs->{core}->{sid};
                my $core_conn_id = $cs->{core}->{conn_id};
                $prefix = $prefix // '';

                # send NOTICE to client
                $heap->{ircd}->send_notice($core_sid, $core_conn_id, $message, $prefix);

                # log to core log
                core_log_info($message, $prefix);
            },

            darkchannel_JOIN => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $join_keyid, $new_channel) = @_[ARG0, ARG1, ARG2, ARG3];
                my $core_sid = dc_session_data_get($sid, 'core', 'sid');
                my $core_conn_id = dc_session_data_get($sid, 'core', 'conn_id');
                my $key_id = dc_session_data_get($sid, 'key_id');

                if ($new_channel) {
                    # send JOIN to client
                    $heap->{ircd}->send_join($core_sid, $core_conn_id, $channel);
                    # send RPL_NOTOPIC
                    $heap->{ircd}->send_rpl_notopic($core_sid, $core_conn_id, $channel);
                    # send RPL_NAMEREPLY
                    $heap->{ircd}->send_rpl_names($core_sid, $core_conn_id, { $channel => $join_keyid });
                }
                else {
                    foreach my $id (@{ $join_keyid }) {
                        # send JOIN to client
                        $heap->{ircd}->send_join($core_sid, $core_conn_id, $channel, $id);
                    }
                }
            },

            darkchannel_PART => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $part_keyid, $channel_parted) = @_[ARG0, ARG1, ARG2, ARG3];
                my $core_sid = dc_session_data_get($sid, 'core', 'sid');
                my $core_conn_id = dc_session_data_get($sid, 'core', 'conn_id');

                # send PART to client
                $heap->{ircd}->send_part($core_sid, $core_conn_id, $channel, $part_keyid);
            },

            darkchannel_MESSAGE => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $key_id, $message) = @_[ARG0, ARG1, ARG2, ARG3];
                my $core_sid = dc_session_data_get($sid, 'core', 'sid');
                my $core_conn_id = dc_session_data_get($sid, 'core', 'conn_id');

                # send PRIVMSG to client channel
                $heap->{ircd}->send_privmsg($core_sid, $core_conn_id, $channel, $key_id, $message);
            },
        },

        heap => {
            ircd => $core_ircd,
        },
    );

    return ($session_id, $alias_listener);
}

sub dc_poe_core_ircd_initialize()
{
    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client::Core::IRCListener");

    return 1;
}

1;
