#
# Dark Channel Client Conf Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Conf;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Tie::IxHash;
use Config::General qw(ParseConfig SaveConfig SaveConfigString);

use DarkChannel::Node::Client::Conf::Defaults;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $conf);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( $CONF dc_conf_initialize dc_conf_shutdown );

# config to export
our $CONF = {};

# storage directory
my $STORAGE_DIR = undef;

# default options for Config::General
my %DEFAULT_OPT = (
    -Tie => "Tie::IxHash",
    -DefaultConfig => \%CONF_DEFAULT,
    -LowerCaseNames => 1,
    -AutoTrue => 1,
    -MergeDuplicateOptions => 1,
    -MergeDuplicateBlocks => 1,
    -ForceArray => 0,
    -SplitPolicy => 'custom',
    -SplitDelimiter => '\s*=\s*',
    -StoreDelimiter => ' = ',
    -UTF8 => 1,
);


#
# load config file
#
sub dc_conf_load()
{
    # config file path & storage mode
    my $storage_mode = 0700;
    my $file = $STORAGE_DIR . '/'. $CONF_DEFAULT_NAME;

    # create storage dir if not existing
    if (! -d $STORAGE_DIR) {
        File::Path::make_path($STORAGE_DIR, { verbose => 0, mode => $storage_mode })
            or confess("failed to create folder " . $STORAGE_DIR . "!");
    }

    # touch config file if it does not exist to make Config::General is happy
    # who does not like the file to not exist when initializing with defaults
    my $empty = 0;
    if (! -r $file) {
        my $F;
        close($F) if (open($F, ">$file"));
        $empty = 1;
    }

    # load and parse config file using Config::General
    tie my %conf, "Tie::IxHash";
    my $genconf = Config::General->new(
        -ConfigFile => $file,
        %DEFAULT_OPT,
    );
    %conf = $genconf->getall();
    $CONF = $conf{client};

    # store default configuration in config file if config file does not exist
    dc_conf_save() if ($empty);

    return;
}

#
# save to config file
#
sub dc_conf_save()
{
    # config file path & storage mode
    my $storage_mode = 0700;
    my $file = $STORAGE_DIR . '/'. $CONF_DEFAULT_NAME;

    # create storage dir if not existing
    if (! -d $STORAGE_DIR) {
        File::Path::make_path($STORAGE_DIR, { verbose => 0, mode => $storage_mode })
            or confess("failed to create folder " . $STORAGE_DIR . "!");
    }

    # touch config file if it does not exist
    if (! -r $file) {
        if (open(my $F, ">$file"))
        {
            close($F);
        }
    }

    # save config file using Config::General
    tie my %conf, "Tie::IxHash";
    my $conf = \%conf;
    $conf->{client} = $CONF;

    my $genconf = Config::General->new(
        -ConfigHash => $conf,
        %DEFAULT_OPT,
    );
    $genconf->save_file($file);

    return;
}

sub dc_conf_initialize($)
{
    # remember storage directory
    $STORAGE_DIR = shift // $CONF_DEFAULT_DIR;

    # load and remember conf for export
    dc_conf_load();

    return $STORAGE_DIR;
}

sub dc_conf_shutdown()
{
    dc_conf_save();
}

1;
