#
# Dark Channel Client Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Getopt::Long;
use Tie::IxHash;

use DarkChannel::Version;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;

use DarkChannel::Proto::Client;
use DarkChannel::Proto::Client::Request;
use DarkChannel::Proto::Client::Response;
use DarkChannel::Proto::Client::Interpreter;

use DarkChannel::Node::Client::Conf;
use DarkChannel::Node::Client::Term;
use DarkChannel::Node::Client::Core::IRCListener;
use DarkChannel::Node::Client::SignalHandler;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our @ISA = qw( Exporter );
our @EXPORT = qw( dc_client_initialize
                  dc_client_run
                  dc_client_shutdown );
our @EXPORT_OK = qw();

# command line argument hash
my $ARGS = {
    help        => 0,
    version     => 0,
    no_mouse    => 0,
    no_color    => 0,
    no_terminal => 0,
    no_irc_core => 0,
    dump_warn   => 0,
    sshtunnel_host => undef,
    sshtunnel_port => undef,
    sshtunnel_user => undef,
    sshtunnel_bind => undef,
    storage_dir => undef,
};

my $SSHTUNNEL_PID = undef;

sub dc_client_run()
{
    # create protocol interpreter session
    dc_client_interpreter_spawn('ChannelServer-Connector')
        or confess("Failed to create client interpreter session");

    # create client signal handler session
    dc_poe_signalhandler_spawn()
        or confess("Failed to create client signal handler session");

    # create terminal session if enabled
    my $alias_terminal;
    if ($CONF->{settings}->{enable_terminal}) {
        unless($ARGS->{no_terminal}) {
            $alias_terminal = dc_poe_term_spawn()
                or confess("Failed to create terminal session");
        }
    }

    # create core ircd listener session if enabled
    if ($CONF->{settings}->{enable_ircd_core}) {
        # try to start IRC core listener if not disabled on command line
        unless($ARGS->{no_irc_core}) {
            my ($sid, $alias) = dc_poe_core_ircd_spawn($alias_terminal);
            confess("Failed to create core ircd listener session") unless($sid);
        }
    }

    # run POE::Kernel
    if (($CONF->{settings}->{enable_terminal}) && not($ARGS->{no_terminal})) {
        # POE::Kernel->run() sessions while handling Curses correctly (our own mainloop() function)
        dc_poe_term_run();
    }
    else {
        # run POE
        POE::Kernel->run();
    }

    return;
}

sub dc_client_print_help()
{
    my $app = $CONF->{product_name} . ' Chat ' . $CONF->{service_name};
    my %flags = ();
    tie %flags, 'Tie::IxHash';

    %flags = (
        'storage-dir=<dir>'     => { desc => 'client data storage folder', short =>'d' },
        'sshtunnel-host=<host>' => { desc => 'ssh remote host for irc core listening port projection' },
        'sshtunnel-port=<port>' => { desc => 'ssh remote port for irc core listening port projection' },
        'sshtunnel-user=<user>' => { desc => 'ssh remote user for irc core listening port projection' },
        'sshtunnel-bind=<user>' => { desc => 'ssh remote bind address for irc core listening port projection' },
        'no-mouse'              => { desc => 'disable mouse support' },
        'no-color'              => { desc => 'disable color support' },
        'no-terminal'           => { desc => 'disable curses terminal support' },
        'no-irc-core'           => { desc => 'disable irc core support' },
        'warn'                  => { desc => 'show warnings on client exit' },
        'version'               => { desc => 'show client version', short => 'v' },
        'help'                  => { desc => 'show this help text', short => 'h' },
    );

    sub fmt($$)
    {
        my ($str, $len) = @_;
        $str .= ' ' while (length($str) < $len);
        return $str;
    }

    print "\n" . $app . ' v' . sprintf("%.2f", $VERSION) . " (" . $CONF->{client_type} . ")\n\n";
    print "Usage: " . $0 . " ";
    print "[--" . $_ . "] " for (keys %flags);
    print "\n\n";
    for (keys %flags) {
        my $short = $flags{$_}->{short} ? ' (-' . $flags{$_}->{short} . ')' : '';
        my $desc = $flags{$_}->{desc};
        printf("%s--%s%s\n", fmt('', 4), fmt($_ . $short, 30), $desc);
    }
    print "\n";
    exit(1);
}

sub dc_client_print_version()
{
    my $app = $CONF->{product_name} . ' Chat ' . $CONF->{service_name};
    print $app . ' ' . sprintf("%.2f", $VERSION) . " (" . $CONF->{client_type} . ")\n";
    exit(1);
}

sub dc_client_parse_argv()
{
    my $res = GetOptions(
        "storage-dir=s"     => \$ARGS->{storage_dir},
        "d=s"               => \$ARGS->{storage_dir},
        "sshtunnel-host=s"  => \$ARGS->{sshtunnel_host},
        "sshtunnel-port=s"  => \$ARGS->{sshtunnel_port},
        "sshtunnel-user=s"  => \$ARGS->{sshtunnel_user},
        "sshtunnel-bind=s"  => \$ARGS->{sshtunnel_bind},
        "no-mouse+"         => \$ARGS->{no_mouse},
        "no-color+"         => \$ARGS->{no_color},
        "no-terminal+"      => \$ARGS->{no_terminal},
        "no-irc-core+"      => \$ARGS->{no_irc_core},
        "warn+"             => \$ARGS->{dump_warn},
        "version+"          => \$ARGS->{version},
        "v+"                => \$ARGS->{version},
        "help+"             => \$ARGS->{help},
        "h+"                => \$ARGS->{help},
    );
}

sub dc_client_sshtunnel_initialize()
{
    my $host = ($ARGS->{sshtunnel_host}) // '127.0.0.1';
    my $port = $ARGS->{sshtunnel_port};
    my $user = ($ARGS->{sshtunnel_user}) // 0;
    my $bind = ($ARGS->{sshtunnel_bind}) // 0;

    if ((not $port) && (($host eq '127.0.0.1') || ($host eq 'localhost'))) {
        confess("need specified port for ssh tunnel host " . $host . "!")
    }
    else {
        $port = $CONF->{node}->{core}->{ircd}->{_default}->{bind_port} // 6667;
    }

    # build ssh command
    my $sap = $host . ':' . $port;
    my $bind_host = $CONF->{node}->{core}->{ircd}->{_default}->{bind_host};
    my $bind_port = $CONF->{node}->{core}->{ircd}->{_default}->{bind_port};
    my $port_spec = ($bind ? $bind . ':' : '') . $port . ':' . $bind_host . ':' . $bind_port;
    my $host_spec = ($user ? ($user . '@' . $host) : $host);
    my $ssh_cmd = 'ssh ' . $host_spec . ' -R ' . $port_spec . ' -N';
    dc_log_dbg("executing '" . $ssh_cmd . "'", 'Client: SSH Tunnel');

    # fork a seperate process to start the sshtunnel command
    $SSHTUNNEL_PID = fork;
    if (!defined $SSHTUNNEL_PID) {
        confess("dc_client_sshtunnel_setup(): cannot fork: $!");
    }
    # client process
    elsif ($SSHTUNNEL_PID == 0) {
        # redirect STDOUT to STDERR
        close(STDOUT);
        open(STDOUT,">&STDERR");
        # should never return
        exec($ssh_cmd);
    }
    dc_log_dbg('forked ssh process with pid ' . $SSHTUNNEL_PID, 'Client: SSH Tunnel');
    dc_log_info('listening on ' . $sap);
}

sub dc_client_sshtunnel_shutdown()
{
    dc_log_dbg("sending kill signal to pid " . $SSHTUNNEL_PID, 'Client: SSH Tunnel');
    kill(9, $SSHTUNNEL_PID);
    waitpid($SSHTUNNEL_PID, 0);
}



sub dc_client_initialize()
{
    # parse command line arguments
    dc_client_parse_argv();

    # initialize conf
    my $storage_dir = dc_conf_initialize($ARGS->{storage_dir});

    # show help or version if requested
    dc_client_print_help() if ($ARGS->{help});
    dc_client_print_version() if ($ARGS->{version});

    # initialize log module
    dc_log_initialize($CONF, $storage_dir, 1, 1, '%H:%M:%S') unless ($ARGS->{no_terminal});
    dc_log_initialize($CONF, $storage_dir, 0, 0, '%H:%M') if ($ARGS->{no_terminal});

    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client");

    # initialize crypto subsystems
    crypt_base_initialize($CONF, $storage_dir)
        or confess("Failed to initialize DarkChannel::Crypt::Base!");

    # initialize client request/response subsystem
    dc_client_request_initialize()
        or confess("Failed to initialize DarkChannel::Proto::Client::Request!");
    dc_client_response_initialize()
        or confess("Failed to initialize DarkChannel::Proto::Client::Response!");
    dc_client_interpreter_initialize()
        or confess("Failed to initialize DarkChannel::Proto::Client::Interpreter!");

    # initialize poe signal handler subsystem
    dc_poe_signalhandler_initialize()
        or confess("Failed to initialize DarkChannel::Node::Client::SignalHandler!");

    # initialize poe term client subsystem
    unless ($ARGS->{no_terminal}) {
        my $mouse_support = $ARGS->{no_mouse} ? 0 : 1;
        my $color_support = $ARGS->{no_color} ? 0 : 1;
        my $irc_core_support = $ARGS->{no_irc_core} ? 0 : 1;
        dc_poe_term_initialize($mouse_support, $color_support, $irc_core_support)
            or confess("Failed to initialize DarkChannel::Node::Client::Term!");
    }

    # initialize poe core ircd listener subsystem
    if ($CONF->{settings}->{enable_ircd_core}) {
        unless ($ARGS->{no_irc_core}) {
            dc_poe_core_ircd_initialize()
                or confess("Failed to initialize DarkChannel::Node::Client::Core::IRCListener!");
        }
    }

    # create ssh tunnel to listener if requested
    dc_client_sshtunnel_initialize() if (($ARGS->{sshtunnel_host}) || ($ARGS->{sshtunnel_port}));

    return 1;
}

sub dc_client_shutdown()
{
    crypt_base_shutdown();
    dc_client_sshtunnel_shutdown() if (($ARGS->{sshtunnel_host}) || ($ARGS->{sshtunnel_port}));
    dc_conf_shutdown();
    dc_log_shutdown($ARGS->{dump_warn});
}

1;
