#
# Dark Channel ChannelServer Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::ChannelServer;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Getopt::Long;
use POSIX qw(strftime);
use Tie::IxHash;

use DarkChannel::Version;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::V1;
use DarkChannel::Proto::Server;
use DarkChannel::Proto::Server::Request;
use DarkChannel::Proto::Server::Response;
use DarkChannel::Proto::Server::Interpreter;

use DarkChannel::Node::ChannelServer::Conf;
use DarkChannel::Node::ChannelServer::SignalHandler;

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Component::Server::TCP);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our @ISA = qw( Exporter );
our @EXPORT = qw( dc_channelserver_initialize
                  dc_channelserver_run
                  dc_channelserver_shutdown
                  dc_channelserver_sshtunnel_pid );
our @EXPORT_OK = qw();

# command line argument hash
my $ARGS = {
    help           => 0,
    version        => 0,
    no_detach      => 0,
    dump_warn      => 0,
    sshtunnel_host => undef,
    sshtunnel_port => undef,
    sshtunnel_user => undef,
    sshtunnel_bind => undef,
    storage_dir    => undef,
};

my $SSHTUNNEL_PID = undef;

sub dc_channelserver_run()
{
    # create protocol interpreter session
    dc_server_interpreter_spawn('ChannelServer-Listener')
        or confess("Failed to create server interpreter session!");

    # create signal handler session
    dc_poe_signalhandler_spawn()
        or confess("Failed to create client listener session!");

    # create client listener session
    my $key_id = $CONF->{node}->{channelserver}->{_default}->{key_id};      # keyid to use for this channel server
    my $listener_host = $CONF->{node}->{channelserver}->{_default}->{host};
    my $listener_port = $CONF->{node}->{channelserver}->{_default}->{port};
    dc_poe_server_spawn($listener_host, $listener_port, $key_id)
        or confess("Failed to create protocol server listener session!");

    # run POE
    POE::Kernel->run();
}

sub dc_channelserver_print_help()
{
    my $app = $CONF->{product_name} . ' Chat ' . $CONF->{service_name};
    my %flags = ();
    tie %flags, 'Tie::IxHash';

    %flags = (
        'storage-dir=<dir>'     => { desc => 'channel server data storage folder', short =>'d' },
        'sshtunnel-host=<host>' => { desc => 'ssh remote host for channel server listening port projection' },
        'sshtunnel-port=<port>' => { desc => 'ssh remote port for channel server listening port projection' },
        'sshtunnel-user=<user>' => { desc => 'ssh remote user for channel server listening port projection' },
        'sshtunnel-bind=<user>' => { desc => 'ssh remote bind address for channel server listening port projection' },
        'no-detach'             => { desc => 'do not detach from controlling tty' },
        'no-color'              => { desc => 'disable color support' },
        'warn'                  => { desc => 'show warnings on channel server exit' },
        'help'                  => { desc => 'show this help text', short => 'h' },
    );

    sub fmt($$)
    {
        my ($str, $len) = @_;
        $str .= ' ' while (length($str) < $len);
        return $str;
    }

    print "\n" . $app . ' v' . sprintf("%.2f", $VERSION) . " " . $CONF->{server_type} . "\n\n";
    print "Usage: " . $0 . " ";
    print "[--" . $_ . "] " for (keys %flags);
    print "\n\n";
    for (keys %flags) {
        my $short = $flags{$_}->{short} ? ' (-' . $flags{$_}->{short} . ')' : '';
        my $desc = $flags{$_}->{desc};
        printf("%s--%s%s\n", fmt('', 4), fmt($_ . $short, 30), $desc);
    }
    print "\n";
    exit(1);
}

sub dc_channelserver_print_version()
{
    my $app = $CONF->{product_name} . ' Chat ' . $CONF->{service_name};
    print $app . ' ' . sprintf("%.2f", $VERSION) . " (" . $CONF->{server_type} . ")\n";
    exit(1);
}

sub dc_channelserver_parse_argv()
{
    my $res = GetOptions(
        "storage-dir=s"     => \$ARGS->{storage_dir},
        "d=s"               => \$ARGS->{storage_dir},
        "sshtunnel-host=s"  => \$ARGS->{sshtunnel_host},
        "sshtunnel-port=s"  => \$ARGS->{sshtunnel_port},
        "sshtunnel-user=s"  => \$ARGS->{sshtunnel_user},
        "sshtunnel-bind=s"  => \$ARGS->{sshtunnel_bind},
        "no-detach"         => \$ARGS->{no_detach},
        "warn"              => \$ARGS->{dump_warn},
        "version+"          => \$ARGS->{version},
        "v+"                => \$ARGS->{version},
        "help+"             => \$ARGS->{help},
        "h"                 => \$ARGS->{help},
    );
}

sub dc_channelserver_sshtunnel_initialize()
{
    my $host = ($ARGS->{sshtunnel_host}) // '127.0.0.1';
    my $port = $ARGS->{sshtunnel_port};
    my $user = ($ARGS->{sshtunnel_user}) // 0;
    my $bind = ($ARGS->{sshtunnel_bind}) // 0;

    if ((not $port) && (($host eq '127.0.0.1') || ($host eq 'localhost'))) {
        confess("need specified port for ssh tunnel host " . $host . "!")
    }
    else {
        $port = $CONF->{node}->{channelserver}->{_default}->{port} // 26667;
    }

    # build ssh command
    my $sap = $host . ':' . $port;
    my $bind_host = $CONF->{node}->{channelserver}->{_default}->{host};
    my $bind_port = $CONF->{node}->{channelserver}->{_default}->{port};
    my $port_spec = ($bind ? $bind . ':' : '') . $port . ':' . $bind_host . ':' . $bind_port;
    my $host_spec = ($user ? ($user . '@' . $host) : $host);
    my $ssh_cmd = 'ssh ' . $host_spec . ' -R ' . $port_spec . ' -N';
    dc_log_dbg("executing '" . $ssh_cmd . "'", 'ChannelServer: SSH Tunnel');

    # fork a seperate process to start the sshtunnel command
    $SSHTUNNEL_PID = fork;
    if (!defined $SSHTUNNEL_PID) {
        confess("dc_channelserver_sshtunnel_setup(): cannot fork: $!");
    }
    # client process
    elsif ($SSHTUNNEL_PID == 0) {
        # redirect STDOUT to STDERR
        close(STDOUT);
        open(STDOUT,">&STDERR");
        # should never return
        exec($ssh_cmd);
    }
    dc_log_dbg('forked ssh process with pid ' . $SSHTUNNEL_PID, 'ChannelServer: SSH Tunnel');
    dc_log_info('listening on ' . $sap);
}

sub dc_channelserver_sshtunnel_shutdown()
{
    if ($SSHTUNNEL_PID) {
        dc_log_dbg("sending kill signal to pid " . $SSHTUNNEL_PID, 'ChannelServer: SSH Tunnel');
        kill(9, $SSHTUNNEL_PID);
        waitpid($SSHTUNNEL_PID, 0);
    }
}

sub dc_channelserver_initialize()
{
    # parse command line arguments
    dc_channelserver_parse_argv();

    # initialize conf first
    my $storage_dir = dc_conf_initialize($ARGS->{storage_dir});

    # show help or version if requested
    dc_channelserver_print_help() if ($ARGS->{help});
    dc_channelserver_print_version() if ($ARGS->{version});

    # initialize log module
    dc_log_initialize($CONF, $storage_dir, 0, $ARGS->{no_detach} ? 0 : 1);

    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Server");

    # initialize cyrpto base subsystem
    crypt_base_initialize($CONF, $storage_dir)
        or confess("Failed to initialize DarkChannel::Crypto::Base!");

    # initialize the signal handler
    dc_poe_signalhandler_initialize()
        or confess("Failed to initialize DarkChannel::Node::ChannelServer::SignalHandler!");

    # initialize channel server request/response subsystem
    dc_server_request_initialize()
        or confess("Failed to initialize DarkChannel::Proto::ChannelServer::Request!");
    dc_server_response_initialize()
        or confess("Failed to initialize DarkChannel::Proto::ChannelServer::Response!");
    dc_server_interpreter_initialize()
        or confess("Failed to initialize DarkChannel::Proto::ChannelServer::Interpreter!");

    # initialize server listener
    dc_poe_server_initialize()
        or confess("Failed to initialize DarkChannel::Proto::ClientListener!");

    # create ssh tunnel to listener if requested
    dc_channelserver_sshtunnel_initialize() if (($ARGS->{sshtunnel_host}) || ($ARGS->{sshtunnel_port}));

    return 1;
}

sub dc_channelserver_shutdown()
{
    crypt_base_shutdown();
    dc_channelserver_sshtunnel_shutdown() if (($ARGS->{sshtunnel_host}) || ($ARGS->{sshtunnel_port}));
    dc_conf_shutdown();
    dc_log_shutdown($ARGS->{dump_warn});
}

1;
