#
# Dark Channel Crypto Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Crypt::GPG;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use File::Path;
use POSIX qw( strftime );
use IPC::Run3;
use Time::HiRes qw( time );
use FileHandle;
use File::Which;

use DarkChannel::Utils::Log;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( crypt_gpg_initialize
                  crypt_gpg_shutdown
                  crypt_gpg_agent_start
                  crypt_gpg_agent_stop
                  crypt_gpg_agent_check
                  crypt_gpg_agent_setup_passphrase
                  crypt_gpg_key_generate
                  crypt_gpg_key_information
                  crypt_gpg_key_information_dump
                  crypt_gpg_key_information_dump_hash
                  crypt_gpg_key_exists
                  crypt_gpg_key_sign
                  crypt_gpg_key_trust
                  crypt_gpg_key_export
                  crypt_gpg_key_import
                  crypt_gpg_key_inspect
                  crypt_gpg_sign_data
                  crypt_gpg_encrypt_data
                  crypt_gpg_decrypt_data
                  crypt_gpg_verify_data );

# configuration passed to us on initialize()
my $CONF;
my $STORAGE_DIR;
my $GPG_AGENT_PID;

# gpg default flags
my $GPG_FLAGS = [];
my $GPG_KEYRING_FLAGS = [];

#
# override standard log functions and add prefix
#

sub crypt_gpg_log($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_info($msg, $prefix);
}

sub crypt_gpg_log_info($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_info($msg, $prefix);
}

sub crypt_gpg_log_warn($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_warn($msg, $prefix);
}

sub crypt_gpg_log_err($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_err($msg, $prefix);
}

sub crypt_gpg_log_crit($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_crit($msg, $prefix);
}

sub crypt_gpg_log_dbg($;$)
{
    my $msg = shift;
    my $prefix = shift;
    $prefix = $prefix ? 'GPG: ' . $prefix : 'GPG';
    dc_log_dbg($msg, $prefix) if ($CONF->{log}->{log_dbg_gpg});
}

#
# returns the path to the GPG storage folder
#
sub crypt_gpg_storage_folder()
{
    return $STORAGE_DIR . '/gpg';
}

#
# returns fully qualified ids for key specification
#
# supports: 'name'                        -> name@fqdn
#           'name@host                    -> name@host
#           '0xID'                        -> 0xID
#           [ name1, name2@host, 0xID]    -> name1@fqdn name2@host 0xID ...
#
# returns: one id string of id's concatenated by $separator
#
sub crypt_gpg_key_idfq($;$)
{
    my $id = shift;
    my $separator = shift // ' ';

    my $fqdn = `hostname -f`; chomp($fqdn);
    my @result = ();

    return @result if (not $id);

    if (ref($id) eq 'ARRAY') {
        foreach my $i (@{ $id })
        {
            push(@result, $i) if ($i =~ /^0x/); # keyid
            push(@result, ($_ =~ m{@}) ? $i : $i . '@' . $fqdn) unless ($i =~ /^0x/); # email
        }
    }
    elsif (scalar($id)) {
        push(@result, $id) if ($id =~ /^0x/); # keyid
        push(@result, ($id =~ m{@}) ? $id : $id . '@' . $fqdn) unless ($id =~ /^0x/); # email
    }
    return @result;
}

#
# returns fully qualified ids for key specification
#
# returns: a string
#
sub crypt_gpg_key_idfq_str($;@)
{
    my $id = shift;
    my @id = shift // crypt_gpg_key_idfq($id);
    my @id_formated = ();

    return '' if (not $id);

    push(@id_formated, ($_ =~ /^0x/) ? $_ : '<' . $_ . '>') for (@id);

    my $ids = (($#id gt 0) ? "identities" : "identity") . " " . join(', ', @id_formated);
    return $ids;
}

#
# returns an array of --keyring <x> und --secret-keyring <x> flags for each
# filename in the given array reference while checking for the files.
# the first file is used always, otherwise we can not create keyrings.
#
sub crypt_gpg_key_file_flags($)
{
    my $files = shift;
    my $storage = crypt_gpg_storage_folder();

    $files = [ $files ] unless (ref($files));

    confess('$files is empty!') unless($files);
    confess('$files is no an array reference!') if (ref($files) ne 'ARRAY');

    my @flags = ();
    for my $file (@{ $files }) {
        my $pubfile = $file . '.pub';
        my $pubpath = $storage . '/' . $pubfile;
        my $secfile = $file . '.sec';
        my $secpath = $storage . '/' . $secfile;

        push(@flags, '--keyring' , $pubfile);
        push(@flags, '--secret-keyring' , $secfile);
    }
    return @flags;
}

#
# execute GPG command with default flags$input scalar and
# $output reference with additional $gpg_flags.
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_cmd_execute($gpg_flags, $input, \$output)
#

sub crypt_gpg_cmd_execute($;$$$)
{
    my $out = '';
    my $gpg_flags = shift;
    my $input = shift // '';
    my $output = shift // \$out;
    my $no_keyrings = shift // 0;

    # don't use keyring flags when $no_keyrings is set
    my @keyring_flags = ();
    @keyring_flags = @{ $GPG_KEYRING_FLAGS } unless ($no_keyrings);

    # create gpg command: executable, default flags, keyring flags, passed flags
    my @gpg_cmd = ( $CONF->{gpg}->{gpg_executables}->{gpg}, @{ $GPG_FLAGS }, @keyring_flags, @{ $gpg_flags } );

    # execute command
    crypt_gpg_log_dbg(join (' ', @keyring_flags, @{ $gpg_flags })) if ($CONF->{log}->{log_dbg_gpg_exec});

    dc_log_untie_stderr();
    run3(\@gpg_cmd, \$input, $output, $output);
    dc_log_tie_stderr();

    my $err = ($? >> 8);

    # append result to logile
    my $path = $CONF->{gpg}->{gpg_logfile};
    if ($path) {
        my $gpg_logfile = ($path =~ /^\//) ? $path : $STORAGE_DIR . '/' . $path; # allow absolute and relative
        if (open(FILE, '>>' . $gpg_logfile))
        {
            my $now = strftime "%Y-%m-%d %H:%M", localtime;
            print FILE "======== GPG Startup at " . $now . " =========\n\n";
            print FILE ${$output};
            close(FILE);
        }
    }

    return 0 if ($err);
    return 1;
}

#
# check if a key for the given $id_email exists in the keyring $id_file
# and return it's key id, key len and key date for public and secret key
# as well as the fingerprint of the public key
#
# returns: hash of data on success
#          '' on failure
#
# crypt_gpg_key_fingerprint($id_key)
#

sub crypt_gpg_key_information($;$)
{
    my $id_key = shift;
    my $secret = shift // 0;

    my @id = crypt_gpg_key_idfq($id_key);
    my $ids = crypt_gpg_key_idfq_str($id_key, @id);

    my $output = '';
    my $gpg_type = $secret ? '-K' : '-k';
    my $gpg_flags = [ $gpg_type, '--fingerprint', '--fingerprint', @id ];
    my $ret = crypt_gpg_cmd_execute($gpg_flags, '', \$output);

    crypt_gpg_log_dbg($output, 'Key Information Output') if ($CONF->{log}->{log_dbg_gpg_keyinfo});

    # parse output
    my $result = {};
    my @lines = split(/\n/, $output);
    my $last = undef;
    my $last_type = undef;
    my $last_key = undef;
    my $last_key_type = undef;
    for my $line (@lines)
    {
        # public key
        if ($line =~ /^pub\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)/)
        {
            my ($publen, $pubid, $pubdate) = ($1, $2, $3);
            $result->{pub}->{$pubid} = {
                key_id => $pubid,
                key_length => $publen,
                key_date => $pubdate,
            };
            $last = $result->{pub}->{$pubid};
            $last_type = 'pub';
            $last_key = $last;
            $last_key_type = 'pub';
        }
        # secret key
        if ($line =~ /^sec\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)/)
        {
            my ($seclen, $secid, $secdate) = ($1, $2, $3);
            $result->{sec}->{$secid} = {
                key_id => $secid,
                key_length => $seclen,
                key_date => $secdate,
            };
            $last = $result->{sec}->{$secid};
            $last_type = 'sec';
            $last_key = $last;
            $last_key_type = 'pub';
        }
        # append these to last pub/sec key
        if ($line =~ /^uid\s+\[\s*([^\s]+)\s*\] (.*) \((.+)\) <([^\s]+)>/)
        {
            my ($uidtrust, $uidname, $uidcomment, $uidemail) = ($1, $2, $3, $4);
            $last->{uid} = [] if (not $last->{uid});
            push(@{ $last->{uid} },  {
                uid_email => $uidemail,
                uid_name => $uidname,
                uid_comment => $uidcomment,
                uid_trust => $uidtrust,
                 });
        }
        elsif ($line =~ /^uid\s+(.*) \((.+)\) <([^\s]+)>/)
        {
            my ($uidname, $uidcomment, $uidemail) = ($1, $2, $3);
            $last->{uid} = [] if (not $last->{uid});
            push(@{ $last->{uid} },  {
                uid_email => $uidemail,
                uid_name => $uidname,
                uid_comment => $uidcomment,
                 });
        }
        if ($line =~ /^sub\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)/)
        {
            my ($sublen, $subid, $subdate) = ($1, $2, $3);
            $last->{sub}->{$subid} = {
                key_id => $subid,
                key_length => $sublen,
                key_date => $subdate,
            };
            $last_key = $last->{sub}->{$subid};
            $last_type = 'sub';
        }
        if ($line =~ /^ssb\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)/)
        {
            my ($ssblen, $ssbid, $ssbdate) = ($1, $2, $3);
            $last->{ssb}->{$ssbid} = {
                key_id => $ssbid,
                key_length => $ssblen,
                key_date => $ssbdate,
            };
            $last_key = $last->{ssb}->{$ssbid};
            $last_key_type = 'ssb';
        }
        # append these to last key/subkey (every key has a fingerprint)
        if ($line =~ /Key fingerprint = (.+)/)
        {
            my $fingerprint = $1;
            $fingerprint =~ s/\s//g;
            $last_key->{key_fpt} = $fingerprint;
        }
    }

    #crypt_gpg_log_dbg(Dumper($result), "Key Data Dump");

    # check result and return it
    return $result if (%{ $result });

    # return undef otherwise
    crypt_gpg_log("no valid key pair found" . ($ids ? ' for ' . $ids : ''));
    return undef;
}

#
# dump data returned from crypt_gpg_key_information()
#
# returns a multiline string
#
sub crypt_gpg_key_information_dump($)
{
    my $keyinfo = shift;
    my $r = '';

    # print pub keys, sub keys and uids
    for my $keyid (keys %{ $keyinfo->{pub} })
    {
        # ignore key email double entries
        next unless ($keyid =~ /^0x/);

        # print uid
        if ($keyinfo->{pub}->{$keyid}->{uid})
        {
            for my $u (@{ $keyinfo->{pub}->{$keyid}->{uid} }) {
                $r .= "GPG Key [uid]: ".$u->{uid_name}." (".$u->{uid_comment}.
                    ") [".$u->{uid_trust}."] <".$u->{uid_email}.">\n";
            }
        }

        # print pub key
        my $k = $keyinfo->{pub}->{$keyid};
        $r .= "GPG Key [pub]: id=".$k->{key_id}.", length=".$k->{key_length}.
            ", date=".$k->{key_date}."\n";

        # print sub key
        if ($keyinfo->{pub}->{$keyid}->{sub})
        {
            for my $subkeyid (keys %{ $keyinfo->{pub}->{$keyid}->{sub} }) {
                my $s = $keyinfo->{pub}->{$keyid}->{sub}->{$subkeyid};
                $r .= "GPG Key [sub]: id=".$s->{key_id}.", length=".$s->{key_length}.
                    ", date=".$s->{key_date}."\n";
            }
        }
    }
    return $r unless ($keyinfo->{sec});

    # print sec keys, ssb keys and uids
    for my $keyid (keys %{ $keyinfo->{sec} })
    {
        # ignore key email double entries
        next unless ($keyid =~ /^0x/);

        # print uid
        if ($keyinfo->{sec}->{$keyid}->{uid})
        {
            for my $u (@{ $keyinfo->{sec}->{$keyid}->{uid} }) {
                $r .= "GPG Key [uid]: ".$u->{uid_name}." (".$u->{uid_comment}.
                    ") [".$u->{uid_trust}."] <".$u->{uid_email}.">\n";
            }
        }

        # print sec key
        my $k = $keyinfo->{sec}->{$keyid};
        $r .= "GPG Key [sec]: id=".$k->{key_id}.", length=".$k->{key_length}.
            ", date=".$k->{key_date}."\n";

        # print ssb key
        if ($keyinfo->{sec}->{$keyid}->{ssb})
        {
            for my $ssbkeyid (keys %{ $keyinfo->{sec}->{$keyid}->{ssb} }) {
                my $s = $keyinfo->{sec}->{$keyid}->{ssb}->{$ssbkeyid};
                $r .= "GPG Key [ssb]: id=".$s->{key_id}.", length=".$s->{key_length}.
                    ", date=".$s->{key_date}."\n";
            }
        }
    }
    return $r;
}

#
# dump data returned from crypt_gpg_key_information_dump_hash()
#
# returns a multiline string
#
sub crypt_gpg_key_information_dump_hash($)
{
    my $keyinfo = shift;
    my $r = {};

    # print pub keys, sub keys and uids
    for my $keyid (keys %{ $keyinfo->{pub} })
    {
        # ignore key email double entries
        next unless ($keyid =~ /^0x/);

        # print uid
        if ($keyinfo->{pub}->{$keyid}->{uid})
        {
            my $u = (@{ $keyinfo->{pub}->{$keyid}->{uid} })[0];
            my $k = $keyinfo->{pub}->{$keyid};
            my $key = {
                id => $keyid,
                type => 'pub',
                name => $u->{uid_name},
                comment => $u->{uid_comment},
                email => $u->{uid_email},
                trust => $u->{uid_trust},
                date => $k->{key_date},
            };
            $r->{$keyid} = $key;
        }
    }
    return $r unless ($keyinfo->{sec});

    # print sec keys, ssb keys and uids
    for my $keyid (keys %{ $keyinfo->{sec} })
    {
        # ignore key email double entries
        next unless ($keyid =~ /^0x/);

        # print uid
        if ($keyinfo->{sec}->{$keyid}->{uid})
        {
            my $u = (@{ $keyinfo->{sec}->{$keyid}->{uid} })[0];
            my $k = $keyinfo->{sec}->{$keyid};
            my $key = {
                id => $keyid,
                type => 'sec',
                name => $u->{uid_name},
                comment => $u->{uid_comment},
                email => $u->{uid_email},
                trust => $u->{uid_trust},
                date => $k->{key_date},
            };
            $r->{$keyid} = $key;
        }
    }
    return $r;
}

#
# check if a key for the given $id_email exists in the keyrings
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_key_exists($id_key)
#

sub crypt_gpg_key_exists($)
{
    my $id_key = shift;

    return crypt_gpg_key_information($id_key);
}

#
# sign public key for the given $pubkey_id with private key $id_key
# $id_file can be a array reference of file names
#
# returns: output
#

sub crypt_gpg_key_sign($$$;$$$)
{
    my ($id_key, $id_pass, $pubkey_id, $type, $domain, $level) = @_;

    $type = 'sign' unless($type);
    $domain = '' unless($domain);

    my @id = crypt_gpg_key_idfq($id_key);
    my $ids = crypt_gpg_key_idfq_str($id_key, @id);
    my $id = $id[0];

    confess("more than one id_key received for signing!") if ($#id gt 0);
    confess("unknow sign type '" . $type . "' for signing!")
        unless (($type eq 'sign') || ($type eq 'lsign') || ($type eq 'tsign')
                || ($type eq 'nrsign') || ($type eq 'ltsign') || ($type eq 'lnrsign') || ($type eq 'tnrsign'));


    # create gpg flags
    my $gpg_flags = [ '--edit', $pubkey_id, $type, 'save' ];
    unshift(@{ $gpg_flags }, '--enable-special-filenames');
    unshift(@{ $gpg_flags }, '--command-fd', '0');
    unshift(@{ $gpg_flags }, '--default-sig-expire', '1d');
    unshift(@{ $gpg_flags }, '--default-key', $id);
    unshift(@{ $gpg_flags }, '--passphrase', $id_pass) if ($id_pass);

    # IO
    my $input = "y\n";
    my $output = '';

    $input = "2\n" . ($level // 1) . "\n" . $domain . "\ny\n"
        if (($type eq 'tsign') || ($type eq 'ltsign') || ($type eq 'tnrsign'));

    # sign key
    crypt_gpg_log_dbg("signing key (keyid=" . $pubkey_id . ", type=" . $type . ", level=" . ($level // 1) . ")");
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $input, \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Key Sign Result') if ($CONF->{log}->{log_dbg_gpg_key_sign});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to sign key using " . $ids);
        return '';
    }
    crypt_gpg_log("successfully signed key using " . $ids);
    return $output;
}

#
# set trust for a key for the given $id_key
# $id_file can be a array reference of file names
#
# returns: output
#
# crypt_gpg_key_trust($id_key, $trust)
#

sub crypt_gpg_key_trust($$)
{
    my ($id_key, $trust) = @_;

    my @id = crypt_gpg_key_idfq($id_key);
    my $ids = crypt_gpg_key_idfq_str($id_key, @id);
    my $id = $id[0];

    confess("more than one id_key received for signing!") if ($#id gt 0);
    confess("trust is not between 1 and 5!") if (($trust < 1) || ($trust > 5));


    # create gpg flags
    my $gpg_flags = [ '--edit', $id, "trust", "save" ];
    unshift(@{ $gpg_flags }, '--enable-special-filenames');
    unshift(@{ $gpg_flags }, '--command-fd', '0');
    unshift(@{ $gpg_flags }, '--status-fd', '2');

    # set trust
    my $input = $trust . "\ny\n";
    my $output = '';
    crypt_gpg_log_dbg("setting trust for key (keyid=" . $id . ") to level " . $trust);
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $input, \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Key Trust Result') if ($CONF->{log}->{log_dbg_gpg_key_trust});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to set trust for key using " . $ids);
        return '';
    }
    crypt_gpg_log("successfully set trust for key using " . $ids);
    return $output;
}

#
# fetch public key for the given $id_key and return it
#
# returns: public key on success
#          '' on failure
#
# crypt_gpg_key_export($id_key)
#

sub crypt_gpg_key_export($)
{
    my $id_key = shift // confess("crypt_gpg_key_export() called without id_key, that's not what you want!");

    if (ref($id_key) eq 'ARRAY') {
        my @tmp = @{ $id_key };
        confess("crypt_gpg_key_export() called with empty reference, that's not what you want!") unless(@tmp);
    }

    my @id = crypt_gpg_key_idfq($id_key);
    my $ids = crypt_gpg_key_idfq_str($id_key, @id);

    my $gpg_flags = [ '--armor', '--export', @id ];
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, '', \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Export Result') if ($CONF->{log}->{log_dbg_gpg_export});
    if (not $ret)
    {
        crypt_gpg_log("no valid key pair found for " . $ids);
        return '';
    }
    crypt_gpg_log("fetched and returned armored public key for " . $ids);
    return $output;
}

sub crypt_gpg_key_import_parseresult($)
{
    my $output = shift;

    # parse import result
    my $result = {};
    my @lines = split(/\n/, $output);
    my $count = 0;
    for my $line (@lines)
    {
        if ($line =~ /^gpg: pub\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)\s+(.*) \((.+)\) <([^\s]+)>/)
        {
            my ($publen, $pubid, $pubdate, $uidname, $uidcomment, $uidemail) = ($1, $2, $3, $4, $5, $6);
            $result->{$pubid} = {
                key_id => $pubid,
                key_type => 'pub',
                key_length => $publen,
                key_date => $pubdate,
                uid_name => $uidname,
                uid_email => $uidemail,
                uid_comment => $uidcomment,
            };
            $count++;
        }
        if ($line =~ /^gpg: sec\s+([0-9]+.)\/(0x[0-9A-F]+)\s+([0-9]+-[0-9]+-[0-9]+)\s+(.*) \((.+)\) <([^\s]+)>/)
        {
            my ($seclen, $secid, $secdate, $uidname, $uidcomment, $uidemail) = ($1, $2, $3, $4, $5, $6);
            $result->{$secid} = {
                key_id => $secid,
                key_type => 'sec',
                key_length => $seclen,
                key_date => $secdate,
                uid_name => $uidname,
                uid_email => $uidemail,
                uid_comment => $uidcomment,
            };
            $count++;
        }
    }

    return ($count, $result);
}

#
# import public key into the keyring $id_destfile
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_key_import($id_destfile, $pubkey)
#

sub crypt_gpg_key_import($$)
{
    my $id_destfile = shift;
    my $pubkey = shift;

    my $gpg_flags = [ '--primary-keyring', $id_destfile, '-v', '--import' ];
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $pubkey, \$output, 0);

    crypt_gpg_log_dbg($output, 'Import Result') if ($CONF->{log}->{log_dbg_gpg_import});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to import public key to key ring '" . $id_destfile . "'");
        return undef;
    }

    my ($count, $result) = crypt_gpg_key_import_parseresult($output);
    #crypt_gpg_log_dbg(Dumper($result), 'Import Result Dump');

    crypt_gpg_log("successfully imported " . $count . " key(s) to key ring '" . $id_destfile . "'");
    return $result;
}

#
# inspect public key
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_key_inspect($keyid)
#

sub crypt_gpg_key_inspect($)
{
    my $pubkey = shift;

    my $gpg_flags = [ '--dry-run', '-v', '--import' ];
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $pubkey, \$output);

    crypt_gpg_log_dbg($output, 'Inspect Result') if ($CONF->{log}->{log_dbg_gpg_inspect});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to inspect key data");
        return undef;
    }

    my ($count, $result) = crypt_gpg_key_import_parseresult($output);

    #crypt_gpg_log_dbg(Dumper($result), 'Inpect Result Dump');
    crypt_gpg_log('successfully inspected ' . $count . ' key(s)');
    return $result;
}

#
# use key for the given $id_email to sign data
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_sign_data($id_key, $id_pass, $data)
#

sub crypt_gpg_sign_data($$$)
{
    my $id_key = shift;
    my $id_pass = shift;
    my $data = shift;

    my @id = crypt_gpg_key_idfq($id_key);
    my $ids = crypt_gpg_key_idfq_str($id_key, @id);
    my $id = $id[0];

    confess("more than one id_key received for signing!") if ($#id gt 0);

    # XXX: TODO: pass passphrase differently. don't keep it in variables & RAM

    my $gpg_flags = [ '--detach-sign', '--armor' ];
    unshift(@{ $gpg_flags }, '--default-sig-expire', '1d');
    unshift(@{ $gpg_flags }, '--default-key', $id);
    unshift(@{ $gpg_flags }, '--passphrase', $id_pass) if ($id_pass);

    #crypt_gpg_log_dbg("signing data using key identity <" . $id . ">");
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $data, \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Sign Result') if ($CONF->{log}->{log_dbg_gpg_sign});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to sign data using " . $ids);
        return '';
    }
    crypt_gpg_log("successfully signed data using " . $ids);
    return $output;
}

#
# encrypt (and sign if $sign) data for recipients $id_recipient_key with $id_key
#
# returns: encrypted data on success
#          0 on failure
#
# crypt_gpg_encrypt_data($id_key, $id_pass, $id_recipient_key, $data, $sign)
#

sub crypt_gpg_encrypt_data($$$$$)
{
    my ($id_key, $id_pass, $id_recipient_key, $data, $sign) = @_;

    my @id = crypt_gpg_key_idfq($id_recipient_key);
    my $ids = crypt_gpg_key_idfq_str($id_recipient_key, @id);
    my $keys = $id_key;

    $keys = [ $id_key ] if (ref($keys) ne 'ARRAY');

    # XXX: TODO: pass passphrase differently. don't keep it in variables & RAM

    my $gpg_flags = [ '--encrypt', '--armor' ];
    unshift(@{ $gpg_flags }, '--sign') if ($sign);
    unshift(@{ $gpg_flags }, '--default-sig-expire', '1d');
    unshift(@{ $gpg_flags }, '--local-user', $_) for (@{ $keys });
    unshift(@{ $gpg_flags }, '--passphrase', $id_pass) if ($id_pass);
    push(@{ $gpg_flags }, '--recipient', $_) for (@id);

    crypt_gpg_log_dbg("encrypting data using signing key(s) " . join(', ',@{$keys})
                      . " and recipient key(s) " . $ids);
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $data, \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Encrypt Result') if ($CONF->{log}->{log_dbg_gpg_encrypt});
    if (not $ret)
    {
        crypt_gpg_log_err("failed to encrypt data for recipients " . $ids);
        return '';
    }
    crypt_gpg_log("successfully encrypted data for recipients " . $ids);
    return $output;
}

#
#
#
sub crypt_gpg_parse_signature($;$)
{
    my $output = shift;
    my $no_encryption = shift // 0;

    # split into encryption/decrypted/signature and store key/subkey relations
    my @lines = split(/\n/, $output);
    my @encryption = ();
    my @signature = ();
    my $subkeys = {};
    for (0..$#lines) {
        last if ($no_encryption);
        my $l = shift(@lines);
        push(@encryption, $l);
        $subkeys->{$1} = $2 if ($l =~ /^gpg: using subkey (0x[A-F0-9]+) instead of primary key (0x[A-F0-9]+)$/);
        last if ($l =~ /^gpg: original file name=/);
    }
    for (0..$#lines) {
        my $l = pop(@lines);
        unshift(@signature, $l);
        $subkeys->{$1} = $2 if ($l =~ /^gpg: using subkey (0x[A-F0-9]+) instead of primary key (0x[A-F0-9]+)$/);
        last if ($l =~ /^gpg: Signature made /) && (not grep(/^gpg: Signature made /, @lines));
    }
    my @decrypted = @lines;
    my $decrypted = join("\n", @decrypted);

    #crypt_gpg_log_dbg(join("\n", @encryption), "split [encryption]");
    #crypt_gpg_log_dbg(join("\n", @signature), "split [signature]");
    #crypt_gpg_log_dbg(join("\n", @decrypted), "split [decrypted]");

    # parse signature
    my ($sign_algo, $sign_id, $sign_keyid, $sign_subkeyid, $sign_date, $sign_unknown, $sign_keytype);
    my ($signature, $nickname) = (0, 0);

    while (@signature) {
        my $l = shift(@signature);
        my $last = ($#signature < 0) ? 1 : 0;

        $sign_algo = $1 if ($l =~ /^gpg: binary signature, digest algorithm ([^\s]+)$/);
        $sign_id = $1 if ($l =~ /^gpg: Good signature from \"(.*)\"/);
        ($sign_keytype, $sign_subkeyid) = ($1, $2) if ($l =~ /^gpg: \s+using ([^\s]+) key (0x[A-F0-9]+)$/);
        $sign_unknown = 1 if ($l =~ /^gpg: Can't check signature: No public key$/);
        ($sign_subkeyid, $sign_keyid) = ($1, $2)
            if ($l =~ /^gpg: using subkey (0x[A-F0-9]+) instead of primary key (0x[A-F0-9]+)$/);

        if ($last || (($l =~ /^gpg: Signature made (.+)$/) && $sign_date)) {
            # find key_id for subkey_id if not found in signature
            $sign_keyid = $subkeys->{$sign_subkeyid};

            # check validity
            my $sign_state = ($sign_id && $sign_keyid && $sign_subkeyid && $sign_date)
                ? 'valid/known' : ($sign_unknown ? 'valid/unknown' : 'invalid');

            # create signature data hash
            my $sign_data = {
                id    => $sign_id // '',
                key_id => $sign_keyid // '',
                subkey_id => $sign_subkeyid // '',
                date  => $sign_date // '',
                state => $sign_state // '',
            };

            # store signature depending on type
            if ($sign_id) {
                if (($sign_id =~ /\(Client\)/) || ($sign_id =~ /\(Channel Server\)/)) {
                    $signature = $sign_data
                }
                elsif ($sign_id =~ /\(Nickname\)/) {
                    $nickname = $sign_data
                }
                else {
                    confess('received unparsable signature (sign_id=' . $sign_id . ')');
                }
            }
            # store signature when no public key has been found
            elsif ($sign_keyid || $sign_subkeyid) {
                $signature = $sign_data
            }
            # reset parsed data
            ($sign_algo, $sign_id, $sign_keyid, $sign_subkeyid, $sign_date, $sign_unknown, $sign_keytype)
                = (undef, undef, undef, undef, undef, undef, undef);
        }
        $sign_date = $1 if ($l =~ /^gpg: Signature made (.+)$/);
    }

    # generate result hash
    my $result = {
        data => $decrypted // '',
        signature => $signature // { state => 'invalid' },
    };
    $result->{nickname} = $nickname if ($nickname);

    return $result;
}

#
# use key $id_key (and $id_pass for the key) to decrypt data $data
#
# returns: hash of data on success
#          0 on failure
#
# crypt_gpg_decrypt_data($id_key, $id_pass, $data)
#

sub crypt_gpg_decrypt_data($$$)
{
    my ($id_key, $id_pass, $data) = @_;

    # XXX: TODO: pass passphrase differently. don't keep it in variables & RAM

    my $gpg_flags = [ '-v', '--decrypt' ];
    unshift(@{ $gpg_flags }, '--default-key', $id_key);
    unshift(@{ $gpg_flags }, '--passphrase', $id_pass) if ($id_pass);

    crypt_gpg_log_dbg("decrypting data using key " . $id_key);
    my $output = '';
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $data, \$output);

    chomp($output);
    crypt_gpg_log_dbg($output, 'Decrypt Result') if ($CONF->{log}->{log_dbg_gpg_decrypt});
    if (!$ret && !$output)
    {
        crypt_gpg_log_err("gpg failed to decrypt data using " . $id_key);
        return '';
    }

    # parse command output
    my $result = crypt_gpg_parse_signature($output);
    my $decrypted = $result->{data};

    # log verify output
    if ($CONF->{log}->{log_dbg_gpg_verify}) {
        crypt_gpg_log_dbg($output, 'Decrypt Output');
        crypt_gpg_log_dbg('result = ' . Dumper($result), 'Decrypt Result');
    }

    if ($decrypted) {
        my $sign_state = $result->{signature}->{state};
        my $sign_keyid = $result->{signature}->{key_id};

        crypt_gpg_log("successfully decrypted valid data signed by " . $sign_state . " entity "
                      . $sign_keyid . " using key " . $id_key);
        return $result;
    }
    crypt_gpg_log_err("failed to decrypt and parse data using key " . $id_key . ":\n"
                  . 'result = ' . Dumper($result) . "\n");
    return '';
}

#
# use key $id_key to verify data given a signture
#
# returns: (1, $output)  on success
#          (0, $output) on failure
#
# crypt_gpg_verify_data($id_key, $signature, $data)
#

sub crypt_gpg_verify_data($$$)
{
    my ($id_key, $signature, $data) = @_;
    my $output = '';

    # create pipe pair and make sure file descriptor does not get closed on exec/fork
    pipe my ($sig_in, $sig_out);
    fcntl($sig_in, F_SETFD, 0);

    # send signature using signature out file descriptor and close it afterwards
    $signature .= "\n\n";
    print $sig_out $signature;
    close($sig_out);

    # use signature in file descriptor for signature and fd 0 (stdin) for data
    my $fd = fileno($sig_in);
    my $gpg_flags = [ '--enable-special-filenames', '-v', '--verify', '--', '-&' . $fd, '-' ];
    unshift(@{ $gpg_flags }, '--default-key', $id_key);

    # call gpg
    #crypt_gpg_log_dbg("verifying data using key " . $id_key);
    my $ret = crypt_gpg_cmd_execute($gpg_flags, $data, \$output);
    chomp($output);

    # close signature in file desciptor
    close($sig_in);

    # parse command output
    my $result = crypt_gpg_parse_signature($output, 1);
    my $sig = $result->{signature};
    my $sign_state = $sig->{state};

    # log verify output
    if ($CONF->{log}->{log_dbg_gpg_verify}) {
        crypt_gpg_log_dbg($output, 'Verify Result');
        crypt_gpg_log_dbg('result = ' . Dumper($result), 'Verify Result');
    }

    # log result
    if ($sign_state eq 'valid/known') {
        crypt_gpg_log("successfully verified data signed by known entity");
    }
    elsif ($sign_state eq 'valid/unknown') {
        crypt_gpg_log("successfully verified data signed by unknown entity");
    }
    else {
        crypt_gpg_log_err("failed to verify data using key " . $id_key);
    }
    return $result;
}

sub crypt_gpg_agent_start()
{
    my @cmd = ( $CONF->{gpg}->{gpg_executables}->{gpg_agent} );
    my $homedir = crypt_gpg_storage_folder();
    my $cache_ttl_default = 60 * 30; # [s] 30min
    my $cache_ttl_max = 999999;      # [s] 11.57 days
    my @flags = (
        '--daemon',
        '--sh',
        '--homedir', $homedir,
        '--default-cache-ttl', $cache_ttl_default,
        '--max-cache-ttl', $cache_ttl_max,
        '--allow-preset-passphrase',
    );
    my ($in, $out, $err) = ('', '', '');

    # add flags to command
    push(@cmd, @flags);

    # clear gpg-agent data from environment to make sure to start a new gpg-agent
    delete $ENV{'GPG_AGENT_INFO'};

    crypt_gpg_log_info('starting gpg-agent (flags=' . join (' ', @flags) . ')', 'Agent');

    dc_log_untie_stderr();
    run3(\@cmd, \$in, \$out, \$err);
    dc_log_tie_stderr();

    my $err_exit = ($? >> 8);

    chomp($out);
    if ($out =~ /^GPG_AGENT_INFO=(.+):(\d+):(\d+); export GPG_AGENT_INFO;$/) {
        my ($agent_socket, $agent_pid, $agent_count) = ($1, $2, $3);

        # store pid for stop
        crypt_gpg_log_dbg('started gpg-agent (pid=' . $agent_pid . ', socket=' . $agent_socket . ')', 'Agent');
        $GPG_AGENT_PID = $agent_pid;

        # set environement so that gpg tools use gpg-agent
        my $info = $agent_socket . ':' . $agent_pid . ':' . $agent_count;

        crypt_gpg_log_dbg('setting gpg-agent environment (GPG_AGENT_INFO=' . $info . ')', 'Agent');
        $ENV{'GPG_AGENT_INFO'} = $info;
    }
    else {
        crypt_gpg_log_crit('ATTENTION: failed to start gpg-agent!', 'Agent');
        crypt_gpg_log_crit('application will not be able to cache passphrases!', 'Agent');
    }

    return;
}

sub crypt_gpg_agent_stop()
{
    crypt_gpg_log_dbg('stopping gpg-agent (pid=' . $GPG_AGENT_PID . ')', 'Agent');
    kill(2, $GPG_AGENT_PID) if (defined($GPG_AGENT_PID));
    return;
}

sub crypt_gpg_agent_check()
{
    my @cmd = ( $CONF->{gpg}->{gpg_executables}->{gpg_connect_agent} );
    my $in = "NOP\n";
    my $out = '';

    crypt_gpg_log_dbg('checking gpg-agent functionality by executing a NOP', 'Agent-Check');

    dc_log_untie_stderr();
    run3(\@cmd, \$in, \$out, \$out);
    dc_log_tie_stderr();

    chomp($out);
    crypt_gpg_log_dbg("gpg-agent check returned '" . $out . "'", 'Agent-Check');

    return ($out eq 'OK');
}

#
# setup gpg-agent to cache the given passphrase for the given keyid
# using the external gpg too gpg-preset-passphrase
#
# returns: 0 if passphrase has been cached by gpg-agent
#          $key_passphrase on failure
#
# crypt_gpg_agent_setup_passphrase($key_id, \@key_fpr, $key_passphrase)
#
sub crypt_gpg_agent_setup_passphrase($$$)
{
    my ($key_id, $key_fpr, $key_passphrase) = @_;
    my $preset = $CONF->{gpg}->{gpg_executables}->{gpg_preset_passphrase};

    # preset passphrase for all $fpr
    my @preset_cmd = ( $preset, '-v', '--preset' );
    my $input = $key_passphrase . "\n";
    my $output = '';
    foreach my $fpr (@{ $key_fpr }) {
        # execute command gpg-preset-passphrase
        my @cmd = ( @preset_cmd, $fpr );
        my ($out, $err) = ('', '');

        crypt_gpg_log_dbg(join(' ', @cmd), 'Agent-Preset') if ($CONF->{log}->{log_dbg_gpg_exec});
        dc_log_untie_stderr();
        run3(\@cmd, \$input, \$out, \$err);
        dc_log_tie_stderr();
        crypt_gpg_log_err("Output: '" . $out . "'", 'Agent-Preset:') if ($out);
        crypt_gpg_log_info("successfully preset passphrase for key (fingerprint=" . $fpr . ")", 'Agent-Preset')
            unless($out);

        $output .= $out;
    }

    # no output means the passphrase has been cached
    return 0 unless($output);

    # alert user of gpg-agent preset passphrase failure
    crypt_gpg_log_crit("ATTENTION: failed to cache passphrase for key " . $key_id . " using 'gpg-preset-passphrase'!", 'Agent-Preset');
    crypt_gpg_log_crit("ATTENTION: falling back to using gpg with --passphrase. this is inherently insecure!");
    crypt_gpg_log_crit("HINT: make sure your gpg-agent.conf contains a 'allow-preset-passphrase' statement,");
    crypt_gpg_log_crit("HINT: or start gpg-agent with the --allow-preset-passphrase flag!");

    # return passphrase, caching has failed
    return $key_passphrase
}

#
# generate a new gpg key in the given key ring $id_file
#
# returns: keyinfo has on success
#          undef on failure
#
# crypt_gpg_key_generate($id_file, $id_name, $id_user, $id_host, $id_comment, $id_passphrase)
#
sub crypt_gpg_key_generate($$$$$$)
{
    my $id_file = shift;
    my $id_name = shift;
    my $id_user = shift;
    my $id_host = shift;
    my $id_comment = shift;
    my $id_passphrase = shift;

    confess('$id_file may not be a reference in crypt_gpg_key_generate()!') if(ref($id_file));

    my $now = strftime "%Y-%m-%d", localtime;
    my $fqdn = `hostname -f`; chomp($fqdn);

    my $gpg_passphrase = $id_passphrase ? ("Passphrase: " . $id_passphrase) : "%ask-passphrase";

    my $gpg_keylen = $CONF->{gpg}->{gpg_keylength};
    my @gpg_keyring = crypt_gpg_key_file_flags($id_file);
    my $gpg_flags = [ @gpg_keyring, '-v', '--gen-key' ];
    my $gpg_keydef =
        "%echo Generating a RSA key pair...
        Key-Type: RSA
        Key-Length: " . $gpg_keylen . "
        Subkey-Type: RSA
        Subkey-Length: " . $gpg_keylen . "
        Name-Real: " . $id_name . "
        Name-Comment: " . $id_comment . "
        Name-Email: " . $id_user . '@' . ($id_host // $fqdn) . "
        Creation-Date: " . $now . "
        Expire-Date: 0
        " . $gpg_passphrase ."
        # do a commit
        %commit
        %echo RSA key generation finished.";

    crypt_gpg_log_info("creating RSA key pair for identity '" . $id_name . " (" . $id_comment . ") <" . $id_user . '@' . $fqdn . ">'");
    crypt_gpg_log_info("PLEASE GENERATE ENTROPY! KEY GENERATION CAN TAKE 5 MINUTES AND MORE OTHERWISE!");
    crypt_gpg_log_dbg("generating new RSA key pair with key length " . $gpg_keylen . " in key ring '" . $id_file . "'");

    crypt_gpg_log_dbg($gpg_keydef, 'Key Generation Definition') if ($CONF->{log}->{log_dbg_gpg_keygen});

    # generate new GPG Key and measure time to generate key
    my $output = '';
    my $time_start = time;
    my $result_keygen = crypt_gpg_cmd_execute($gpg_flags, $gpg_keydef, \$output, 1);
    my $time_elapsed = time - $time_start;

    # parse command output for keyid
    my @lines = split(/\n/, $output);
    my ($keyid, $keytype, $ok) = (undef, undef, 0);
    for my $line (@lines) {
        ($keyid,$keytype) = ($2,$1) if ($line =~ /^gpg: (.+) signature from: \"(0x[0-9A-F]+) \[/);
        last if ($keyid);
    }

    # warn to use a entropy gathering daemon if key generation took to longer than 10s
    if ($time_elapsed > 10) {
        crypt_gpg_log_warn('KEY GENERATION TOOK SUBSTANTION TIME!');
        crypt_gpg_log_warn('CONSIDER USING A ENTROPY GATHERING DAEMON TO SPEED UP KEY GENERATION!');
    }

    # show result in log
    crypt_gpg_log_dbg($output, 'Key Generation Result') if ($CONF->{log}->{log_dbg_gpg_keygen});
    crypt_gpg_log("generated new " . $keytype . " key pair with key id " . $keyid . " in "
                  . sprintf("%.2f", $time_elapsed) . " seconds");

    # return key information
    return crypt_gpg_key_exists($keyid);
}

#
# initialize DarkChannel::Crypt::GPG subsystem
#
# returns: 1 on success
#          0 on failure
#
# crypt_gpg_initialize($CONF)
#
sub crypt_gpg_initialize($$$)
{
    $CONF = shift;
    $STORAGE_DIR = shift;
    my $keyrings = shift;

    # create and initialize storage folder
    my $mode = $CONF->{gpg}->{gpg_storage_mode};
    my $storage = crypt_gpg_storage_folder();

    # fetch GPG binaries from configuration
    my $gpg_executable = $CONF->{gpg}->{gpg_executables}->{gpg};
    my $gpg_agent = $CONF->{gpg}->{gpg_executables}->{gpg_agent};
    my $gpg_connect_agent = $CONF->{gpg}->{gpg_executables}->{gpg_connect_agent};

    # check GPG binary and search for it if not found
    $gpg_executable = which('gpg2') unless (-x "$gpg_executable");
    $gpg_executable = which('gnupg2') unless (-x "$gpg_executable");
    $gpg_executable = which('gpg') unless (-x "$gpg_executable");
    $gpg_executable = which('gnupg') unless (-x "$gpg_executable");

    # check GPG Agent binary and search for it if not found
    $gpg_agent = which('gpg-agent') unless (-x "$gpg_agent");

    # check GPG Connect Agent binary and search for it if not found
    $gpg_connect_agent = which('gpg-agent') unless (-x "$gpg_connect_agent");

    # store found GPG binaries in configuration
    $CONF->{gpg}->{gpg_executables}->{gpg} = $gpg_executable;
    $CONF->{gpg}->{gpg_executables}->{gpg_agent} = $gpg_agent;
    $CONF->{gpg}->{gpg_executables}->{gpg_connect_agent} = $gpg_connect_agent;

    # default gpg flags we always use
    push(@{ $GPG_FLAGS }, '--batch');
    push(@{ $GPG_FLAGS }, '--homedir', $storage);
    push(@{ $GPG_FLAGS }, '--no-default-keyring');
    push(@{ $GPG_FLAGS }, '--keyid-format', '0xlong');
    push(@{ $GPG_FLAGS }, '--trust-model', 'pgp');
    push(@{ $GPG_FLAGS }, '--no-emit-version');

    # initialize GPG framework
    dc_log_dbg("initializing DarkChannel::Crypt::GPG (gpg=" . $gpg_executable
               . ", defaults='" . join(' ', @{ $GPG_FLAGS }) . "')");

    # initialize storage
    if (! -d $storage)
    {
        crypt_gpg_log("creating gpg storage folder " . $storage . " (mode=" . $mode . ")");
        File::Path::make_path($storage, { verbose => 0, mode => $mode }) or return 0;
    }

    # initialize keyrings
    my $output = '';
    foreach my $kring (@{ $keyrings }) {
        my @gpg_keyring = crypt_gpg_key_file_flags($kring);
        my $gpg_flags = [ @gpg_keyring, '--import' ];
        crypt_gpg_cmd_execute($gpg_flags, '', \$output, 0);
    }

    # add gpg keyring flags
    push(@{ $GPG_KEYRING_FLAGS }, crypt_gpg_key_file_flags($keyrings));

    # start DarkChannel's own gpg-agent
    crypt_gpg_agent_start();

    return 1;
}

#
# shutdown subsystem
#
sub crypt_gpg_shutdown()
{
    # stop DarkChannel's own gpg-agent
    crypt_gpg_agent_stop();
}

1;
