#
# Dark Channel Crypto Abstraction Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Crypt::Base;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Crypt::Digest::SHA256 qw( sha256 sha256_hex sha256_b64 sha256_b64u );
use Crypt::Digest::SHA512 qw( sha512 sha512_hex sha512_b64 sha512_b64u );

use Crypt::Eksblowfish::Bcrypt qw(bcrypt_hash en_base64 de_base64 bcrypt);

use DarkChannel::Utils::Log;
use DarkChannel::Crypt::GPG;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( crypt_base_initialize
                  crypt_base_shutdown
                  crypt_base_check_cryptosystems
                  crypt_base_setup_cryptosystems
                  crypt_base_check_nickname
                  crypt_base_setup_nickname
                  crypt_base_key_information
                  crypt_base_key_information_str
                  crypt_base_key_information_hash
                  crypt_base_key_exists
                  crypt_base_key_inspect
                  crypt_base_key_sign
                  crypt_base_key_import
                  crypt_base_key_export
                  crypt_base_key_export_pub
                  crypt_base_data_get
                  crypt_base_data_sign
                  crypt_base_data_encrypt
                  crypt_base_data_decrypt
                  crypt_base_data_verify
                  crypt_base_data_hash_b64
                  crypt_base_data_bcrypt );

# configuration passed to us on initialize()
my $CONF;
my $STORAGE_DIR;

# base crypto framework state
my $CRYPT = {};

# base defaults
my $BASE_DEFAULT;

#
# base default config
#
sub crypt_base_defaults()
{
    my $defaults = undef;

    $defaults =  {
        key_file => 'dcd',
        key_user => 'dcd',
        key_name => 'DarkChannel Channel Server',
        key_comment => 'Channel Server',
        nick_file => 'dcd-nickname',
        nick_name => 'DarkChannel Nickname',
        nick_comment => 'Nickname',
    } if ($CONF->{service_name} eq 'Channel Server');

    $defaults = {
        key_file => 'dcc',
        key_user => 'dcc',
        key_name => 'DarkChannel Client',
        key_comment => 'Client',
        nick_file => 'dcc-nickname',
        nick_name => 'DarkChannel Nickname',
        nick_comment => 'Nickname',
    } if ($CONF->{service_name} eq 'Client');

    confess("Unknown service_name '" . $CONF->{service_name} . "' in DarkChannel::Crypt::Base!")
        unless ($defaults);

    return $defaults;
}

#
# gets data of crypto base state
#
# $val = crypt_base_data_get($key1, $key2)
# $val = crypt_base_data_get($key1, $key2, $key3)
#
sub crypt_base_data_get(;$$$$)
{
    my $key1 = shift // undef;
    my $key2 = shift // 0;
    my $key3 = shift // 0;
    my $key4 = shift // 0;

    # XXX: TODO: switch depending on choosen crypto cipher suite
    my $base = 'gpg';

    return $CRYPT->{$base} if (not $key1);
    return $CRYPT->{$base}->{$key1} if (not $key2);
    return $CRYPT->{$base}->{$key1}->{$key2} if (not $key3);
    return $CRYPT->{$base}->{$key1}->{$key2}->{$key3} if (not $key4);
    return $CRYPT->{$base}->{$key1}->{$key2}->{$key3}->{$key4};
}

#
# hash given data and return it base 64 encoded
#
# returns: hashed data
#
# crypt_base_data_hash_b64($data)
#
sub crypt_base_data_hash_b64($)
{
    my $data = shift;
    return sha256_b64($data);
}

#
# bcrypt password
#
# note: feed hashed password as settings arguments
# note: set setting to undef to generate a fresh hash of password
#
sub crypt_base_data_bcrypt($$;$$)
{
    my ($password, $settings, $salt, $rounds) = @_;
    my $service = $CONF->{product_name} . ' / ' .$CONF->{service_name};
    my $key = $service . $password;

    # generate salt if none given
    unless($salt) {
        my @octet = ('0'..'9');
        $salt .= $octet[rand(@octet)] foreach(0..15);
    }

    # defaults
    $rounds = $rounds // 14;
    $settings = $settings // '$2a$' . $rounds . '$' . en_base64($salt);
    confess('rounds to big!') if ($rounds > 99);

    # use bcrypt() inside eval to make sure we handly wrong settings
    my $hash;
    eval {
        use Crypt::Eksblowfish::Bcrypt qw(bcrypt);
        $hash = bcrypt($key, $settings);
    };

    return $hash;
}

#
# sign given data using default key
#
# returns: detached signature on success
#          '' on failure
#
# crypt_gpg_sign_data($data, $key_id)
#
sub crypt_base_data_sign($$)
{
    my $data = shift;
    my $key_id = shift;
    my $key_passphrase = $CRYPT->{gpg}->{pub}->{$key_id}->{key_passphrase} // $CONF->{gpg}->{gpg_passphrase};

    # XXX: TODO: switch depending on choosen crypto cipher suite

    # sign data
    my $signature = crypt_gpg_sign_data($key_id, $key_passphrase, $data);
    chomp($signature) if ($signature);

    return $signature;
}

#
# encrypt given data for given key_id recipients
#
# returns: encrypted data on success
#          '' on failure
#
# crypt_base_data_encrypt($data, $key_id, $recipient_key_id)
#
sub crypt_base_data_encrypt($$$)
{
    my ($data, $key_id, $recipient_key_id) = @_;
    my $key_passphrase = 0;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    # if scalar key_id, allow passphrase feature, otherwise use agent
    if (ref($key_id) ne 'ARRAY') {
        $key_passphrase = $CRYPT->{gpg}->{pub}->{$key_id}->{key_passphrase} // $CONF->{gpg}->{gpg_passphrase};
    }

    # encrypt and sign data
    my $encrypted = crypt_gpg_encrypt_data($key_id, $key_passphrase, $recipient_key_id, $data, 1);
    chomp($encrypted) if ($encrypted);

    return $encrypted;
}

#
# decypt given data
#
# returns: decrypted data on success
#          '' on failure
#
# crypt_base_data_decrypt($data, $key_id)
#
sub crypt_base_data_decrypt($$)
{
    my ($data, $key_id) = @_;
    my $key_passphrase = $CRYPT->{gpg}->{pub}->{$key_id}->{key_passphrase} // $CONF->{gpg}->{gpg_passphrase};

    # XXX: TODO: switch depending on choosen crypto cipher suite

    # decrypt data
    my $decrypted = crypt_gpg_decrypt_data($key_id, $key_passphrase, $data);

    return $decrypted;
}

#
# verify data given a detached signture
#
# returns: (1, $output) on success
#          (0, $output) on failure
#
# crypt_base_data_verify($signature, $data, $key_id)
#
sub crypt_base_data_verify($$$)
{
    my ($signature, $data, $key_id) = @_;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    # verify data
    my $verify = crypt_gpg_verify_data($key_id, $signature, $data);
    $signature = $verify->{signature} if ($verify->{signature});

    # eliminate --trust-model always warning, we always use it
    #$output =~ s/gpg: WARNING: Using untrusted key!\n//g;

    return $signature;
}

#
# sign key
#
# $type may be 'local' (local signature) or 'local-trusted' (introducer signature, not exportable),
# $type may be 'exportable' (normal signature) or 'exportable-trusted' (introducer signature, exportable),
#
# returns: $output on success
#          undef on failure
#
# crypt_base_key_sign($key_id, $pubkey_id, $type [, $domain, $level])
#
sub crypt_base_key_sign($$$;$$)
{
    my ($key_id, $pubkey_id, $type, $domain, $level) = @_;
    my $key_passphrase = $CRYPT->{gpg}->{pub}->{$key_id}->{key_passphrase} // $CONF->{gpg}->{gpg_passphrase};
    my $output;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    dc_log_info("signing key material (keyid=" . $key_id . ", target_keyid=" . $pubkey_id . "')", 'Identity');

    # sign pubkey (trusted -> ltsign, otherwise -> sign)
    if ($type eq 'local-trusted') {
        $output = crypt_gpg_key_sign($key_id, $key_passphrase, $pubkey_id, 'ltsign', $domain, $level);
    }
    elsif ($type eq 'local') {
        $output = crypt_gpg_key_sign($key_id, $key_passphrase, $pubkey_id, 'lsign');
    }
    elsif ($type eq 'exportable-trusted') {
        $output = crypt_gpg_key_sign($key_id, $key_passphrase, $pubkey_id, 'tsign');
    }
    elsif ($type eq 'exportable') {
        $output = crypt_gpg_key_sign($key_id, $key_passphrase, $pubkey_id);
    }
    else {
        confess("unknown signature type '" . $type . "'!");
    }

    # log result
    dc_log_dbg($output, 'Key Sign Result') if ($CONF->{log}->{log_dbg_gpg_key_sign});

    # return on failure to sign
    return '' unless($output);

    # export signed pubkey
    my $pubkey_signed = crypt_gpg_key_export($pubkey_id)
        or confess("failed to fetch GPG public key " . $pubkey_id . " using DarkChannel::Crypt::GPG!");

    # update public key in $CRYPT
    $CRYPT->{gpg}->{pub}->{$pubkey_id}->{key_pub} = $pubkey_signed;

    return $pubkey_signed;
}

#
# export single (cached) armored public key
#
# returns: public key on success
#          undef on failure
#
# crypt_base_key_export_pub($id_key)
#
sub crypt_base_key_export_pub($)
{
    # XXX: TODO: switch depending on choosen crypto cipher suite

    my $key_id = shift;

    confess("crypt_gpg_key_export_pub() called with reference, that's not what you want!")
        if (ref($key_id) eq 'ARRAY');

    return $CRYPT->{gpg}->{pub}->{$key_id}->{key_pub}
}

#
# export one or more key(s) specified in $key_id which may be a array reference
#
# returns: $output on success
#          undef on failure
#
# crypt_base_key_export($key_id)
#
sub crypt_base_key_export($)
{
    my $key_id = shift;
    my $output = '';

    # XXX: TODO: switch depending on choosen crypto cipher suite

    return crypt_gpg_key_export($key_id);
}

#
# import key
#
# returns: $output on success
#          undef on failure
#
# crypt_base_key_import($pubkey [, $role])
#
sub crypt_base_key_import($;$)
{
    my $pubkey = shift;
    my $role = shift // 'CLIENT';
    my $result = undef;
    my $client_file = $BASE_DEFAULT->{key_file};
    my $nick_file = $BASE_DEFAULT->{nick_file};
    my $dest_file;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    confess("crypt_base_key_export() called without sensible role '$role'!") unless($role);

    # dest_file is the key_ring for import
    $dest_file = $client_file if ($role eq 'CLIENT');
    $dest_file = $nick_file if ($role eq 'NICKNAME');

    # import key
    return crypt_gpg_key_import($dest_file . '.pub', $pubkey);
}

#
# check for (cached) key existance and load key information into $CRYPT if not already loaded
#
# returns: 1 on success
#          0 on failure
#
# crypt_base_key_exists($key_id)
#
sub crypt_base_key_exists($)
{
    my $key_id = shift;
    my $key_check = ((exists($CRYPT->{gpg}->{pub}) && exists($CRYPT->{gpg}->{pub}->{$key_id})
                      && exists($CRYPT->{gpg}->{pub}->{$key_id}->{key_id}))
                     || (exists($CRYPT->{gpg}->{sec}) && exists($CRYPT->{gpg}->{sec}->{$key_id})
                         && exists($CRYPT->{gpg}->{sec}->{$key_id}->{key_id})))
        ? 1 : 0;

    print STDERR ">> key_exists($key_id): $key_check\n";

    unless($key_check) {
        my $keyinfo = crypt_gpg_key_exists($key_id);

        print STDERR ">> keyinfo=" . Dumper($keyinfo);

        return 0 unless($keyinfo);

        my $key_type = (exists($keyinfo->{sec}) ? 'sec' : 'pub');
        $CRYPT->{gpg}->{$key_type}->{$key_id} = $keyinfo->{$key_type}->{$key_id}
            if (exists($keyinfo->{$key_type}->{$key_id}->{key_id}));

        return ($keyinfo ? 1 : 0);
    }
    return 1;
}

#
#
# inspect key
#
# returns: $result hash on success
#          undef on failure
#
# crypt_base_key_inspect($pubkey)
#
sub crypt_base_key_inspect($)
{
    my $pubkey = shift;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    return crypt_gpg_key_inspect($pubkey);
}

#
# key information dump
#
# returns: string on success
#          undef on failure
#
# crypt_base_key_information_str($key_id [, $key_type ])
#
sub crypt_base_key_information_str($;$)
{
    my $key_id = shift;
    my $key_type = shift // 0;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    my $keyinfo = crypt_gpg_key_information($key_id, $key_type);
    return crypt_gpg_key_information_dump($keyinfo);
}

#
# key information dump
#
# returns: array of hashes on success
#          undef on failure
#
# crypt_base_key_information_hash($key_id [, $key_type])
#
sub crypt_base_key_information_hash($;$)
{
    my $key_id = shift;
    my $key_type = shift // 0;

    # XXX: TODO: switch depending on choosen crypto cipher suite

    my $keyinfo = crypt_gpg_key_information($key_id, $key_type);
    return crypt_gpg_key_information_dump_hash($keyinfo);
}

#
# setup key for usage with given passphrase
#
# uses a key material agent for passphrase caching if key_storage is set
# otherwise just setup the passphrase in CRYPT without caching agent (server operation)
#
# crypt_base_setup_passphrase($key_id, $key_passphrase, $key_storage)
#
sub crypt_base_setup_passphrase($$$)
{
    my ($key_id, $key_passphrase, $key_storage) = @_;
    my $passphrase = 0;

    if ($key_passphrase) {
        if ($key_storage) {
            my @key_fpr = ();

            # try to store the given passphrase in gpg-agent. this needs gpg-agent started with
            # --allow-preset-passphrase or an entry 'allow-preset-passphrase' in ~/.gnupg/gpg-agent.conf.
            # if gpg-preset-passphrase fails, we will fall back to using the --passphrase option of gpg.
            # this is inherently unsafe and should not be done on a multi user machine!

            # find fingerprint for pub key
            my $key = $CRYPT->{gpg}->{pub}->{$key_id};
            push(@key_fpr, $key->{key_fpt});

            # find fingerprint for all sub keys
            my $sub_keys = $CRYPT->{gpg}->{pub}->{$key_id}->{sub};
            push(@key_fpr, $sub_keys->{$_}->{key_fpt}) for (keys %{ $sub_keys });

            # setup agent and get 0 in return on success and passphrase in return on failure
            $passphrase = crypt_gpg_agent_setup_passphrase($key_id, \@key_fpr, $key_passphrase);
        }
        else {
            # just store the passphrase in our crypto structure and use it directly
            # this case happens on servers where the config file passphrase gets used
            $passphrase = $key_passphrase;
        }
    }

    # use gpg-agent? (eg. $passphrase == 0)
    if ((not $passphrase) && ($passphrase == 0)) {
        # check if gpg-agent is useable
        $passphrase = undef unless(crypt_gpg_agent_check());
    }

    # store passphrase to use (undef -> default pw, 0 -> gpg-agent, otherwise -> pw)
    $CRYPT->{gpg}->{pub}->{$key_id}->{key_passphrase} = $passphrase;
    return;
}

#
# check supported DarkChannel::Crypt::Base crypto systems
#
# returns: 0 if key material needs to be generated
#          1 if no key material needs to generated
#
sub crypt_base_check_cryptosystems(;$)
{
    my $key_id = shift // $CONF->{defaults}->{channelserver}->{_default}->{key_id} // 0;

    return 0 unless($key_id);
    return 0 unless (crypt_gpg_key_information($key_id));
    return 1;
}

#
# setup supported DarkChannel::Crypt::Base crypto systems
#
# this includes assuring that needed key material is available
# if it's not, it will get generated
#
# returns: 1 on success
#          0 on failure
#
# crypt_base_setup_cryptosystems()
#
sub crypt_base_setup_cryptosystems($;$$)
{
    my $channelserver = shift // undef; # if no channelserver is given we use the fqdn
    my $key_id = shift // $CONF->{defaults}->{channelserver}->{_default}->{key_id} // 0; # key_id from CONF
    my $key_passphrase = shift;

    # store key persistently if not using default from $CONF
    my $key_storage = $key_passphrase ? 1 : 0;
    $key_passphrase = $key_passphrase // $CONF->{gpg}->{gpg_passphrase};

    # base key defaults
    my %D = %{ $BASE_DEFAULT };

    # fetch key information
    my $keyinfo = undef;
    $keyinfo = crypt_gpg_key_exists($key_id) if ($key_id);

    # if key_id is not set or the key_id can not be found
    unless($keyinfo)
    {
        # generate a new GPG key pair
        $keyinfo = crypt_gpg_key_generate($D{key_file},
                                             $D{key_name},
                                             $D{key_user},
                                             $channelserver,
                                             $D{key_comment},
                                             $key_passphrase)
            or confess("failed to generate a GPG key using DarkChannel::Crypt::GPG!");

        # store keyid of first pub key which is the new key
        $key_id = (grep(/^0x[0-9A-F]+/, keys %{ $keyinfo->{pub} }))[0] // confess("no keyid found in keyinfo!");

        # set key trust to ultimate (it is our own key)
        crypt_gpg_key_trust($key_id, 5);
    }

    # export pub key for key_id
    my $key_pub = crypt_gpg_key_export($key_id)
        or confess("failed to fetch GPG public key " . $key_id . " using DarkChannel::Crypt::GPG!");

    # store data (keyinfo and key_pub and key_pass if set) in $CRYPT
    $CRYPT->{gpg}->{pub}->{$key_id} = $keyinfo->{pub}->{$key_id};
    $CRYPT->{gpg}->{pub}->{$key_id}->{key_pub} = $key_pub;

    # setup passphrase
    crypt_base_setup_passphrase($key_id, $key_passphrase, $key_storage);

    # dump show key information in log
    my $info = crypt_gpg_key_information_dump($keyinfo);
    dc_log_info('successfully initialized', 'Cryptosystems Setup');
    dc_log_info($info, 'Key Information');

    # return new key context
    return $key_id;
}

#
# check nickname key material
#
# returns: 0 if key material needs to be generated
#          1 if no key material needs to generated
#
sub crypt_base_check_nickname($$)
{
    my $nickname = shift // confess('crypt_base_check_nickname() called without nickname!');
    my $channelserver = shift // confess('crypt_base_check_nickname() called without channelserver!');

    my $nick_fq = $nickname . '@' . $channelserver;
    my $key_id = $CONF->{node}->{nickname}->{$nick_fq}->{key_id} // 0;

    return 0 unless($key_id);
    return 0 unless (crypt_gpg_key_information($key_id));
    return 1;
}

#
# setup nickname
#
# this includes assuring that needed key material is available
# if it's not, it will get generated
#
# returns: 1 on success
#          0 on failure
#
# crypt_base_setup_cryptosystems($nick, $channelserver [, $key_passphrase])
#
sub crypt_base_setup_nickname($$;$$)
{
    my $nickname = shift // confess('crypt_base_setup_nickname() called without nickname!');
    my $channelserver = shift // confess('crypt_base_setup_nickname() called without channelserver!');
    my $key_passphrase = shift;

    my $nick_fq = $nickname . '@' . $channelserver;
    my $key_id = $CONF->{node}->{nickname}->{$nick_fq}->{key_id} // 0;

    # store key persistently if not using default from $CONF
    my $key_storage = $key_passphrase ? 1 : 0;
    $key_passphrase = $key_passphrase // $CONF->{gpg}->{gpg_passphrase};

    # base key defaults
    my %D = %{ $BASE_DEFAULT };

    # fetch key information
    my $keyinfo = undef;
    $keyinfo = crypt_gpg_key_exists($key_id) if ($key_id);

    # if key_id is not set or the key_id can not be found
    unless($keyinfo)
    {
        # generate a new GPG key pair
        $keyinfo = crypt_gpg_key_generate($D{nick_file},
                                          $D{nick_name},
                                          $nickname,
                                          $channelserver,
                                          $D{nick_comment},
                                          $key_passphrase)
            or confess("failed to generate a GPG key using DarkChannel::Crypt::GPG!");

        # store keyid of first pub key which is the new key
        $key_id = (grep(/^0x[0-9A-F]+/, keys %{ $keyinfo->{pub} }))[0] // confess("no keyid found in keyinfo!");
    }

    # export pub key for key_id
    my $key_pub = crypt_gpg_key_export($key_id)
        or confess("failed to fetch GPG public key " . $key_id . " using DarkChannel::Crypt::GPG!");

    # store data (keyinfo and key_pub and key_pass if set) in $CRYPT
    $CRYPT->{gpg}->{pub}->{$key_id} = $keyinfo->{pub}->{$key_id};
    $CRYPT->{gpg}->{pub}->{$key_id}->{key_pub} = $key_pub;

    # setup passphrase
    crypt_base_setup_passphrase($key_id, $key_passphrase, $key_storage);

    # dump show key information in log
    my $info = crypt_gpg_key_information_dump($keyinfo);
    dc_log_info('successfully initialized', 'Nickname');
    dc_log_info($info, 'Key Information');

    # return new key context
    return $key_id;
}

#
# initialize subsystem
#
sub crypt_base_initialize($$)
{
    $CONF = shift;
    $STORAGE_DIR = shift;
    $BASE_DEFAULT = crypt_base_defaults();

    # all keyrings
    my $keyrings = [ $BASE_DEFAULT->{key_file}, $BASE_DEFAULT->{nick_file} ];

    # initialize logging
    dc_log_dbg("initializing DarkChannel::Crypt::Base (crypto='gpg', keyrings=" . join(',', @$keyrings) . ")");

    # initialize internal state with base defaults
    $CRYPT->{gpg} = { defaults => $BASE_DEFAULT };

    # initialize gpg subsystem
    crypt_gpg_initialize($CONF, $STORAGE_DIR, $keyrings)
        or confess("Failed to initialize DarkChannel::Crypt::GPG!");

    return 1;
}

#
# shutdown subsystem
#
sub crypt_base_shutdown()
{
    # shutdown gpg subsystem
    crypt_gpg_shutdown();
}

1;
